<?php
declare(strict_types=1);
/*
Plugin Name: Yektabot
Description: ساخت و مدیریت خودکار وبهوک‌های ووکامرس برای اتصال به یکتابات.
Version: 1.1.0
Requires at least: 6.0
Requires PHP: 7.4
Tested up to: 6.9
Author: Farshad Ghanbari
Author URI: https://yektabot.com/
Plugin URI: https://yektabot.com/
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: yektabot
*/

namespace Yektabot\WcWebhooks;

if (!defined('ABSPATH')) exit;

final class Plugin
{
    private const OPTION = 'yektabot_wc_webhook';
    private const PAGE = 'yektabot-wc-webhook';
    private const VERSION = '1.1.0';
    private const TOPICS = ['product.created', 'product.updated', 'product.deleted'];
    private const LOG_OPTION = 'yektabot_wc_webhook_log';
    private const CONNECT_OPTION = 'yektabot_wc_webhook_connect';
    private const FULL_SYNC_OPTION = 'yektabot_wc_webhook_full_sync';
    private const TRACK_OPTION = 'yektabot_wc_webhook_track';
    private const INSTALL_ID_OPTION = 'yektabot_wc_install_id';
    private const PAYMENT_SYNC_OPTION = 'yektabot_wc_webhook_payment_sync';
    private const CONNECTION_OK_TRANSIENT = 'yektabot_wc_connection_ok';
    private const CONNECT_URL = 'https://app.yektabot.com/wordpress/woocommerce/connect';
    private const CREDENTIALS_URL = 'https://app.yektabot.com/woocommerce/plugin-credentials';
    private const TRACK_URL = 'https://app.yektabot.com/woocommerce/plugin-events';
    private const REQUIRE_URL = 'https://app.yektabot.com/woocommerce/plugin-requirements';
    private const PAYMENT_CONFIRM_URL = 'https://app.yektabot.com/woocommerce/plugin-payment';
    private const FONT_REMOTE_URL = 'https://yektabot.com/assets/fonts/variable/YekanBakhFaNum-VF.woff';
    private const FONT_CACHE_DIR = 'yektabot-wc-webhooks';
    private const FONT_CACHE_FILE = 'YekanBakhFaNum-VF.woff';
    private const ICON_REMOTE_URL = 'https://yektabot.com/assets/images/icons/favicon.ico';
    private const ICON_CACHE_FILE = 'yektabot-favicon.ico';
    private static ?string $webhookTable = null;
    private static bool $updateRequired = false;
    private static array $updateMeta = [];
    private static bool $requirementsChecked = false;

    public static function boot(): void
    {
        add_action('admin_menu', [self::class, 'menu']);
        add_action('admin_bar_menu', [self::class, 'adminBar'], 90);
        add_action('admin_head', [self::class, 'adminCss'], 90);
        add_action('admin_init', [self::class, 'settings']);
        add_action('admin_init', [self::class, 'checkRequirements'], 5);
        add_action('admin_init', [self::class, 'maybeTrack'], 10);
        add_action('admin_init', [self::class, 'maybeSyncPaymentSettings'], 15);
        add_action('template_redirect', [self::class, 'maybePaymentBridge'], 0);
        add_action('wp_ajax_yektabot_full_sync_start', [self::class, 'ajaxFullSyncStart']);
        add_action('wp_ajax_yektabot_full_sync_step', [self::class, 'ajaxFullSyncStep']);
        // Capture WooCommerce webhook delivery results into our internal "فعالیت‌ها" log.
        add_action('woocommerce_webhook_process_delivery', [self::class, 'onWebhookProcessDelivery'], 10, 2);
        add_action('woocommerce_webhook_delivery', [self::class, 'onWebhookDelivery'], 10, 5);
        // Also capture product events; if Action Scheduler / WP-Cron is broken, fall back to direct delivery.
        add_action('save_post_product', [self::class, 'onProductSaved'], 20, 3);
        add_action('before_delete_post', [self::class, 'onBeforeDeletePost'], 20, 1);
        add_action('update_option_' . self::OPTION, [self::class, 'sync'], 10, 0);
        add_action('admin_init', [self::class, 'maybeSync'], 99);
        register_activation_hook(__FILE__, [self::class, 'activate']);
        register_deactivation_hook(__FILE__, [self::class, 'deactivate']);
    }

    public static function onWebhookProcessDelivery($webhook, $arg): void
    {
        try {
            if (!is_object($webhook) || !method_exists($webhook, 'get_name')) return;
            $name = (string)$webhook->get_name();
            if (!str_starts_with($name, 'Yektabot ')) return;
            $topic = method_exists($webhook, 'get_topic') ? (string)$webhook->get_topic() : '';
            self::log('info', 'webhook_dispatch', ['topic' => $topic, 'name' => $name]);
        } catch (\Throwable $e) {
        }
    }

    public static function onWebhookDelivery($httpArgs, $response, $duration, $arg, $webhookId): void
    {
        try {
            $webhookId = (int)$webhookId;
            $name = '';
            $topic = '';
            $deliveryUrl = '';
            if (class_exists('\WC_Webhook') && $webhookId > 0) {
                $wh = new \WC_Webhook($webhookId);
                $name = (string)$wh->get_name();
                if (!str_starts_with($name, 'Yektabot ')) return;
                $topic = (string)$wh->get_topic();
                $deliveryUrl = (string)$wh->get_delivery_url();
            } else {
                return;
            }
            $code = null;
            $error = null;
            if (is_wp_error($response)) {
                $error = $response->get_error_message();
            } elseif (is_array($response)) {
                $code = function_exists('wp_remote_retrieve_response_code') ? (int)wp_remote_retrieve_response_code($response) : null;
            }
            $level = ($code && $code >= 200 && $code < 300) ? 'info' : 'warn';
            if ($error) $level = 'error';
            self::log($level, 'webhook_delivery', [
                'id' => $webhookId,
                'topic' => $topic,
                'name' => $name,
                'url' => $deliveryUrl,
                'status' => $code,
                'error' => $error,
                'duration_ms' => is_numeric($duration) ? (int)round(((float)$duration) * 1000) : null,
            ]);
        } catch (\Throwable $e) {
        }
    }

    public static function onProductSaved($postId, $post, $update): void
    {
        try {
            $postId = (int)$postId;
            if ($postId <= 0) return;
            if (function_exists('wp_is_post_revision') && wp_is_post_revision($postId)) return;
            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
            if (!is_object($post) || (($post->post_type ?? '') !== 'product')) return;
            $postStatus = (string)($post->post_status ?? '');
            // WooCommerce core: wp_trash_post -> product.deleted, untrashed_post -> product.restored
            $topic = $postStatus === 'trash' ? 'product.deleted' : ($update ? 'product.updated' : 'product.created');
            self::log('info', 'product_event', ['topic' => $topic, 'id' => $postId, 'post_status' => $postStatus, 'fallback' => self::shouldDirectDeliver()]);
            if (self::shouldDirectDeliver()) self::directDeliver($topic, $postId);
        } catch (\Throwable $e) {
        }
    }

    public static function onBeforeDeletePost($postId): void
    {
        try {
            $postId = (int)$postId;
            if ($postId <= 0) return;
            if (function_exists('get_post_type') && get_post_type($postId) !== 'product') return;
            $topic = 'product.deleted';
            self::log('info', 'product_event', ['topic' => $topic, 'id' => $postId, 'fallback' => self::shouldDirectDeliver()]);
            if (self::shouldDirectDeliver()) self::directDeliver($topic, $postId);
        } catch (\Throwable $e) {
        }
    }

    private static function shouldDirectDeliver(): bool
    {
        // If WP-Cron is disabled, Action Scheduler won't run automatically.
        if (defined('DISABLE_WP_CRON') && DISABLE_WP_CRON) return true;
        // If Action Scheduler queue runner isn't scheduled or is overdue, assume deliveries won't happen.
        if (function_exists('wp_next_scheduled')) {
            $next = wp_next_scheduled('action_scheduler_run_queue');
            if (!$next) return true;
            if (is_numeric($next) && (time() - (int)$next) > 600) return true; // overdue by 10 min
        }
        return false;
    }

    private static function directDeliver(string $topic, int $productId): void
    {
        try {
            $opts = self::opts();
            if (empty($opts['url']) || empty($opts['secret'])) {
                self::log('warn', 'direct_delivery_skipped', ['reason' => 'not_connected', 'topic' => $topic, 'id' => $productId]);
                return;
            }
            $url = self::target($opts);
            $body = wp_json_encode(['id' => $productId]);
            if (!$body) $body = '{"id":' . (int)$productId . '}';
            $sig = base64_encode(hash_hmac('sha256', $body, (string)$opts['secret'], true));
            $headers = [
                'Content-Type' => 'application/json',
                'X-WC-Webhook-Topic' => $topic,
                'X-WC-Webhook-Signature' => $sig,
                'X-WC-Webhook-Source' => home_url('/'),
            ];
            $resp = wp_remote_post($url, ['timeout' => 15, 'headers' => $headers, 'body' => $body]);
            if (is_wp_error($resp)) {
                self::log('error', 'direct_delivery', ['topic' => $topic, 'id' => $productId, 'error' => $resp->get_error_message()]);
                return;
            }
            $code = function_exists('wp_remote_retrieve_response_code') ? (int)wp_remote_retrieve_response_code($resp) : null;
            self::log(($code >= 200 && $code < 300) ? 'info' : 'warn', 'direct_delivery', ['topic' => $topic, 'id' => $productId, 'status' => $code, 'url' => $url]);
        } catch (\Throwable $e) {
            self::log('error', 'direct_delivery', ['topic' => $topic, 'id' => $productId, 'error' => $e->getMessage()]);
        }
    }

    public static function menu(): void
    {
        $icon = self::iconUrl() ?: 'dashicons-admin-links';
        add_menu_page('یکتابات', 'یکتابات', 'manage_options', self::PAGE, [self::class, 'page'], $icon, 56);
    }

    private static function iconUrl(): ?string
    {
        if (!function_exists('wp_upload_dir') || !function_exists('wp_safe_remote_get') || !function_exists('wp_remote_retrieve_response_code') || !function_exists('wp_remote_retrieve_body')) return null;
        $upload = wp_upload_dir(null, false);
        if (!is_array($upload) || !empty($upload['error'])) return null;
        $baseDir = rtrim((string)($upload['basedir'] ?? ''), '/');
        $baseUrl = rtrim((string)($upload['baseurl'] ?? ''), '/');
        if (!$baseDir || !$baseUrl) return null;
        $dir = $baseDir . '/' . self::FONT_CACHE_DIR;
        $urlDir = $baseUrl . '/' . self::FONT_CACHE_DIR;
        if (!is_dir($dir)) @wp_mkdir_p($dir);
        $path = $dir . '/' . self::ICON_CACHE_FILE;
        $url = $urlDir . '/' . self::ICON_CACHE_FILE;
        if (file_exists($path) && filesize($path) > 0) return $url;
        $res = wp_safe_remote_get(self::ICON_REMOTE_URL, ['timeout' => 10]);
        $code = wp_remote_retrieve_response_code($res);
        if ($code !== 200) return null;
        $body = wp_remote_retrieve_body($res);
        if (!$body) return null;
        if (@file_put_contents($path, $body) === false) return null;
        return $url;
    }

    public static function adminBar($wp_admin_bar): void
    {
        if (!is_admin() || !current_user_can('manage_options')) return;
        if (!is_object($wp_admin_bar) || !method_exists($wp_admin_bar, 'add_node')) return;
        $icon = self::iconUrl();
        $title = $icon
            ? '<span style="display:inline-flex;align-items:center;gap:6px"><img alt="" src="' . esc_url($icon) . '" style="width:24px;height:24px;border-radius:6px;vertical-align:middle" />یکتابات</span>'
            : 'یکتابات';
        $wp_admin_bar->add_node([
            'id' => 'yektabot',
            'title' => $title,
            'href' => admin_url('admin.php?page=' . self::PAGE),
            'meta' => ['class' => 'yektabot-adminbar'],
        ]);
    }

    public static function adminCss(): void
    {
        if (!is_admin() || !current_user_can('manage_options')) return;
        $slug = 'toplevel_page_' . self::PAGE;
        echo '<style>
        #adminmenu .' . esc_attr($slug) . ' .wp-menu-image{
            background-size:24px 24px !important;
            background-position:center center !important;
            background-repeat:no-repeat !important;
        }
        #adminmenu .' . esc_attr($slug) . ' .wp-menu-image img{width:24px !important;height:24px !important;}
        #adminmenu .' . esc_attr($slug) . ' .wp-menu-image:before{
            font-size:24px !important;
            line-height:24px !important;
            width:24px !important;
            height:24px !important;
        }
        </style>';
    }

    public static function settings(): void
    {
        register_setting(self::PAGE, self::OPTION, ['type' => 'array', 'sanitize_callback' => [self::class, 'sanitize']]);
    }

    public static function checkRequirements(): void
    {
        if (self::$requirementsChecked) return;
        self::$requirementsChecked = true;
        $data = null;
        $body = wp_json_encode([
            'site' => home_url('/'),
            'version' => self::VERSION,
            'install_id' => self::installId(),
        ]);
        $headers = ['Accept' => 'application/json', 'Content-Type' => 'application/json'];
        $opts = self::opts();
        if (!empty($opts['secret']) && $body) {
            $headers['X-Yektabot-Signature'] = base64_encode(hash_hmac('sha256', (string)$body, (string)$opts['secret'], true));
        }
        $resp = wp_remote_post(self::REQUIRE_URL, [
            'timeout' => 10,
            'headers' => $headers,
            'body' => $body,
        ]);
        if (!is_wp_error($resp) && (int)wp_remote_retrieve_response_code($resp) === 200) {
            $decoded = json_decode((string)wp_remote_retrieve_body($resp), true);
            if (is_array($decoded)) $data = $decoded;
        }
        if (!$data) {
            $cache = get_transient('yektabot_wc_requirements');
            if (is_array($cache)) $data = $cache;
        }
        if ($data) set_transient('yektabot_wc_requirements', $data, MINUTE_IN_SECONDS * 5);
        $min = is_array($data) && !empty($data['min_version']) ? (string)$data['min_version'] : self::VERSION;
        $download = is_array($data) && !empty($data['download_url']) ? (string)$data['download_url'] : 'https://downloads.wordpress.org/plugin/yektabot.latest-stable.zip';
        $msg = is_array($data) && !empty($data['message']) ? (string)$data['message'] : 'نسخه افزونه قدیمی است؛ لطفاً به‌روزرسانی کنید.';
        if (version_compare(self::VERSION, $min, '<')) {
            self::$updateRequired = true;
            self::$updateMeta = [
                'min' => $min,
                'download' => $download,
                'message' => $msg,
            ];
        }
    }

    public static function page(): void
    {
        if (!current_user_can('manage_options')) return;
        self::checkRequirements();
        if (self::$updateRequired) {
            self::renderUpdateRequired();
            return;
        }
        if (isset($_POST['yb_action']) && $_POST['yb_action'] === 'connect' && check_admin_referer('yb_connect')) self::startConnect();
        if (isset($_GET['yb_action']) && $_GET['yb_action'] === 'connected') self::handleConnectCallback();
        if (isset($_POST['yb_action']) && $_POST['yb_action'] === 'disconnect' && check_admin_referer('yb_disconnect')) self::disconnect();
        if (isset($_POST['yb_action']) && $_POST['yb_action'] === 'clear_logs' && check_admin_referer('yb_clear_logs')) self::clearLogs();
        if (isset($_POST['yb_action']) && $_POST['yb_action'] === 'toggle_logs' && check_admin_referer('yb_toggle_logs')) self::toggleLogs();
        if (isset($_POST['yb_action']) && $_POST['yb_action'] === 'full_sync' && check_admin_referer('yb_full_sync')) self::startFullSync();
        if (isset($_POST['yb_action']) && $_POST['yb_action'] === 'save_payment' && check_admin_referer('yb_payment')) self::savePaymentSettings();
        $opts = self::opts();
        $logs = array_reverse(self::logs());
        $checks = self::readyDetails();
        $levelMap = ['info' => 'اطلاعات', 'warn' => 'هشدار', 'error' => 'خطا'];
        $messageMap = [
            'synced' => 'همگام‌سازی انجام شد',
            'cleanup' => 'پاک‌سازی انجام شد',
            'connected_from_yektabot' => 'اتصال انجام شد',
            'reconnected_detached_previous' => 'اتصال قبلی قطع شد',
            'full_sync_started' => 'شروع همگام‌سازی کامل محصولات',
            'full_sync_completed' => 'پایان همگام‌سازی کامل محصولات',
            'product_event' => 'رویداد محصول',
            'webhook_dispatch' => 'ارسال وبهوک',
            'webhook_delivery' => 'نتیجه ارسال وبهوک',
            'direct_delivery' => 'ارسال مستقیم',
            'direct_delivery_skipped' => 'ارسال مستقیم انجام نشد',
            'api_keys_created' => 'کلیدهای API ووکامرس ساخته شد',
            'api_keys_failed' => 'خطا در ساخت کلیدهای API ووکامرس',
            'api_keys_skipped' => 'ساخت کلیدهای API انجام نشد',
            'api_keys_pushed' => 'کلیدهای API به یکتابات ارسال شد',
            'api_keys_push_failed' => 'خطا در ارسال کلیدهای API به یکتابات',
            'payment_gateway_request_failed' => 'خطا در اتصال به درگاه پرداخت',
            'payment_gateway_verify_failed' => 'خطا در تایید پرداخت',
            'webhook_created' => 'وبهوک ساخته شد',
            'webhook_updated' => 'وبهوک به‌روزرسانی شد',
            'webhook_failed' => 'خطا در وبهوک',
            'config_missing' => 'تنظیمات ناقص است',
            'woocommerce_inactive' => 'ووکامرس فعال نیست',
            'fallback_db' => 'استفاده از دیتابیس ووکامرس',
            'fallback_cpt' => 'استفاده از پست‌تایپ وبهوک',
            'font_upload_dir_error' => 'خطای مسیر آپلود فونت',
        ];
        $fontUrl = self::fontUrl();
        $fontFace = $fontUrl ? '@font-face{font-family:YekanBakhFaNum_v;src:url("' . esc_url($fontUrl) . '") format("woff-variation"),url("' . esc_url($fontUrl) . '") format("woff");font-display:swap;font-weight:100 900;font-style:normal}' : '';
        echo '<div class="wrap yb-wrap">';
        echo '<style>
        ' . $fontFace . '
        :root{--yb-back-main-3:#f7f8fa;--yb-frame-1:#000;--yb-frame-3:#f6f7fb;--yb-neutral-3:#e6e8ec;--yb-neutral-4:#d6d8e0;--yb-neutral-6:#777e90;--yb-neutral-11:#18191d;--yb-primary-400:#41f187;--yb-primary-600:#0ebe54}
        .yb-wrap{margin:0;max-width:none;width:100%;padding:0 18px 90px;font-family:YekanBakhFaNum_v,IRANSans,system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif !important;color:var(--yb-neutral-11);box-sizing:border-box}
        .yb-wrap *{box-sizing:border-box;font-family:YekanBakhFaNum_v,IRANSans,system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif !important;}
        #wpfooter{position:relative}
        .yb-app{direction:rtl}
        .notice.notice-error{display:none !important;}
        .yb-container{max-width:1180px;margin:0 auto}
        .yb-header{padding-top:22px}
        .yb-header-inner{display:flex;align-items:center;justify-content:space-between;gap:18px;flex-direction:row-reverse}
        .yb-logo{width:110px;height:auto;display:block}
        .yb-hero{display:flex;align-items:stretch;justify-content:space-between;gap:18px;flex-direction:row}
        .yb-brand{width:150px;min-width:150px;max-width:150px;min-height:150px;display:flex;align-items:center;justify-content:center;padding:18px;border-radius:25px;border:1px solid var(--yb-neutral-4);background:linear-gradient(135deg,rgba(65,241,135,.16),rgba(14,190,84,.06));box-shadow:0 18px 45px rgba(0,0,0,.06)}
        .yb-brand img{width:120px;height:auto;display:block;filter:drop-shadow(0 14px 22px rgba(0,0,0,.10))}
        .yb-hero-card{flex:1;min-width:0}
        .yb-menu{display:flex;align-items:center;gap:28px}
        .yb-menu a{font-size:16px;font-weight:700;color:var(--yb-neutral-11);text-decoration:none;transition:.2s}
        .yb-menu a:hover{color:var(--yb-primary-600)}
        .yb-btn{cursor:pointer;border-radius:16px;border:1px solid var(--yb-neutral-6);color:var(--yb-neutral-11);height:56px;display:inline-flex;align-items:center;justify-content:center;font-size:15px;font-weight:900;transition:.2s;padding:0 26px;background:#fff;text-decoration:none}
        .yb-btn:hover{background:#000;color:#fff;border-color:#000}
        .yb-main{padding-top:36px}
        .yb-top{display:flex;align-items:flex-end;justify-content:space-between;gap:16px}
        .yb-title{font-size:42px;line-height:1.2;font-weight:900;margin:0}
        .yb-sub{font-size:14px;color:var(--yb-neutral-6);margin-top:8px}
        .yb-tabs{display:flex;align-items:center;gap:14px}
        .yb-tabs .yb-tab-label{font-size:16px;font-weight:900;color:var(--yb-neutral-11)}
        .yb-tabs a{font-size:16px;font-weight:800;color:var(--yb-neutral-11);text-decoration:none;cursor:pointer;padding-bottom:6px;border-bottom:2px solid transparent}
        .yb-tabs a.yb-active{border-bottom-color:var(--yb-frame-1)}
        .yb-grid{display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:18px;margin-top:24px}
        @media (max-width: 1100px){.yb-title{font-size:34px}.yb-grid{grid-template-columns:1fr}.yb-menu{display:none}}
        @media (max-width: 1100px){.yb-hero{flex-direction:column;gap:14px}.yb-brand{width:100%;min-width:0;max-width:none;min-height:0}}
        @media (max-width: 1100px){.yb-brand img{width:78px}}
        .yb-card{border:1px solid var(--yb-neutral-4);border-radius:25px;background:#fff;padding:22px 26px;position:relative}
        .yb-card-head{display:flex;align-items:flex-start;justify-content:space-between;gap:12px;padding-bottom:18px;margin-bottom:18px;border-bottom:1px solid var(--yb-neutral-4)}
        .yb-card-title{font-size:22px;font-weight:900;margin:0}
        .yb-card-desc{font-size:13px;font-weight:700;color:var(--yb-neutral-6);margin-top:8px}
        .yb-badge{width:64px;height:64px;border-radius:999px;background:var(--yb-primary-400);display:flex;align-items:center;justify-content:center;flex:0 0 64px}
        .yb-badge svg{width:28px;height:28px}
        .yb-actions{display:flex;align-items:center;justify-content:space-between;gap:12px;margin-top:10px}
        .yb-btn-filled{cursor:pointer;background:#000 !important;color:#fff !important;border-color:transparent !important}
        .yb-btn-filled:hover{background:#303030 !important}
        .yb-mini{font-size:12px;font-weight:800;color:var(--yb-neutral-6)}
        .yb-list{display:flex;flex-direction:column;gap:10px}
        .yb-item{display:flex;align-items:center;justify-content:space-between;gap:10px;padding:12px 14px;border-radius:16px;background:var(--yb-frame-3)}
        .yb-item b{font-size:13px;font-weight:900}
        .yb-item span{font-size:12px;font-weight:800;color:var(--yb-neutral-6)}
        .yb-mask{direction:ltr;text-align:left;display:block;font-family:ui-monospace,SFMono-Regular,Menlo,Monaco,Consolas,"Liberation Mono","Courier New",monospace;letter-spacing:.08em}
        .yb-pill{display:inline-flex;align-items:center;justify-content:center;min-width:54px;height:30px;border-radius:999px;font-weight:900;font-size:12px}
        .yb-ok{background:rgba(65,241,135,.25);color:var(--yb-primary-600)}
        .yb-warn{background:rgba(214,216,224,.5);color:var(--yb-neutral-11)}
        .yb-logs{margin-top:24px}
        .yb-log-actions{display:flex;align-items:center;justify-content:space-between;gap:12px;margin-bottom:14px}
        .yb-danger{border-radius:16px;border:1px solid #FBC02D;height:56px;padding:0 18px;background:#FBC02D;font-weight:900;cursor:pointer}
        .yb-danger:hover{background:#000;color:#fff;border-color:#000}
        .yb-switch{display:inline-flex;align-items:center;gap:10px;font-weight:900;cursor:pointer;user-select:none}
        .yb-switch input{position:absolute;opacity:0;width:1px;height:1px;pointer-events:none}
        .yb-switch-ui{position:relative;width:36px;height:20px;border-radius:999px;background:rgba(214,216,224,.8);transition:background .2s ease}
        .yb-switch-ui:after{content:"";position:absolute;top:1px;left:1px;width:16px;height:16px;border-radius:999px;background:#fff;border:1px solid rgba(214,216,224,1);transition:transform .2s ease,border-color .2s ease}
        .yb-switch input:focus-visible + .yb-switch-ui{outline:2px solid rgba(58,91,255,.4);outline-offset:2px}
        .yb-switch input:checked + .yb-switch-ui{background:linear-gradient(90deg,var(--yb-primary-400),var(--yb-primary-600))}
        .yb-switch input:checked + .yb-switch-ui:after{transform:translateX(16px);border-color:#fff}
        .yb-switch:dir(rtl) .yb-switch-ui:after{left:auto;right:1px}
        .yb-switch:dir(rtl) input:checked + .yb-switch-ui:after{transform:translateX(-16px)}
        .yb-table{width:100%;border-collapse:separate;border-spacing:0 10px}
        .yb-table th{text-align:right;color:var(--yb-neutral-6);font-weight:900;font-size:12px;padding:6px}
        .yb-table td{background:var(--yb-frame-3);padding:12px 14px;border-radius:16px;color:var(--yb-neutral-11);font-size:12px;font-weight:800;vertical-align:top}
        .yb-code{direction:ltr;text-align:left;display:block;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;max-width:520px}
        .yb-app .notice{display:none !important}
        </style>';

        echo '<div class="yb-app" dir="rtl">';

        echo '<main class="yb-container yb-main">';
        $conn = self::connectionStatus();
        if (isset($_GET['disconnected']) && (string)$_GET['disconnected'] === '1') {
            echo '<div style="margin:15px 0;padding:14px 16px;border-radius:18px;background:rgba(214,216,224,.55);font-weight:900;color:var(--yb-neutral-11)">اتصال قطع شد.</div>';
        }
        if ($conn === false) {
            echo '<div style="margin:15px 0;padding:14px 16px;border-radius:18px;background:rgba(214,216,224,.55);font-weight:900;color:var(--yb-neutral-11)">اتصال معتبر نیست یا از سمت یکتابات قطع شده است. لطفاً اتصال را مجدد انجام دهید یا قطع اتصال بزنید.</div>';
        }
        if (isset($_GET['connected'])) {
            $ok = (string)$_GET['connected'] === '1';
            $msg = $ok ? 'اتصال انجام شد و تنظیمات خودکار ذخیره شد.' : 'اتصال ناموفق بود.';
            $bg = $ok ? 'rgba(65,241,135,.22)' : 'rgba(214,216,224,.55)';
            $color = $ok ? 'var(--yb-primary-600)' : 'var(--yb-neutral-11)';
            echo '<div style="margin:15px 0;padding:14px 16px;border-radius:18px;background:' . $bg . ';font-weight:900;color:' . $color . '">' . esc_html($msg) . '</div>';
        }
        if (isset($_GET['full_sync'])) {
            $ok = (string)$_GET['full_sync'] === '1';
            $processed = isset($_GET['processed']) ? (int)$_GET['processed'] : 0;
            $updated = isset($_GET['updated']) ? (int)$_GET['updated'] : 0;
            $deleted = isset($_GET['deleted']) ? (int)$_GET['deleted'] : 0;
            $msg = $ok ? ('همگام‌سازی کامل انجام شد. ' . $processed . ' مورد (بروزرسانی: ' . $updated . ' / حذف: ' . $deleted . ')') : 'همگام‌سازی کامل انجام نشد.';
            $bg = $ok ? 'rgba(65,241,135,.22)' : 'rgba(214,216,224,.55)';
            $color = $ok ? 'var(--yb-primary-600)' : 'var(--yb-neutral-11)';
            echo '<div style="margin:15px 0;padding:14px 16px;border-radius:18px;background:' . $bg . ';font-weight:900;color:' . $color . '">' . esc_html($msg) . '</div>';
        }
        if (isset($_GET['payment'])) {
            $ok = (string)$_GET['payment'] === '1';
            $msg = $ok ? 'تنظیمات پرداخت ذخیره شد.' : 'ذخیره تنظیمات پرداخت انجام نشد.';
            $bg = $ok ? 'rgba(65,241,135,.22)' : 'rgba(214,216,224,.55)';
            $color = $ok ? 'var(--yb-primary-600)' : 'var(--yb-neutral-11)';
            echo '<div style="margin:15px 0;padding:14px 16px;border-radius:18px;background:' . $bg . ';font-weight:900;color:' . $color . '">' . esc_html($msg) . '</div>';
        }

        echo '<div class="yb-hero"><div class="yb-brand"><img src="https://yektabot.com/assets/images/yektabot.png" alt="Yektabot"/></div><div class="yb-card yb-hero-card"><h1 class="yb-title">یکپارچه‌سازی هوشمند وردپرس با فروش در اینستاگرام</h1><div class="yb-sub">افزونه وردپرس یکتابات با همگام‌سازی خودکار محصولات، وب‌سایت وردپرسی شما را به یکتابات متصل کرده و امکان شروع فرآیند فروش در اینستاگرام را تنها با ارسال کد محصول توسط مشتری فراهم می‌سازد.</div></div></div>';

        echo '<div class="yb-grid">';
        $maskedSecret = $opts['secret'] ? self::mask((string)$opts['secret'], 4) : '—';
        $maskedAccount = $opts['account_id'] ? self::mask((string)$opts['account_id'], 2) : '—';
        $maskedUrl = $opts['url'] ? self::maskUrl((string)$opts['url'], (string)$opts['account_id']) : '—';

        echo '<div class="yb-card"><div class="yb-card-head"><div><div class="yb-card-title">اتصال</div><div class="yb-card-desc">تنظیمات به‌صورت خودکار از یکتابات دریافت می‌شود</div></div><div class="yb-badge"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none"><path d="M12 3l8 4v6c0 5-3.5 9.4-8 10-4.5-.6-8-5-8-10V7l8-4Z" stroke="white" stroke-width="1.6"/><path d="M8.5 12.2l2.2 2.2L15.8 9.3" stroke="white" stroke-width="1.6" stroke-linecap="round" stroke-linejoin="round"/></svg></div></div>';
        $connected = !empty($opts['url']) && !empty($opts['secret']) && !empty($opts['account_id']);
        $pill = $connected ? ('<span class="yb-pill ' . ($conn === false ? 'yb-warn' : 'yb-ok') . '">' . ($conn === false ? 'نامعتبر' : 'فعال') . '</span>') : '<span class="yb-pill yb-warn">قطع</span>';
        echo '<div class="yb-actions" style="margin-top:-6px;margin-bottom:14px;justify-content:space-between;align-items:center"><div class="yb-mini">' . ($connected ? 'در صورت نیاز می‌توانید اتصال را مجدد انجام دهید' : 'برای دریافت تنظیمات، اتصال خودکار را انجام دهید') . '</div>' . $pill . '</div>';
        echo '<div class="yb-actions" style="margin-top:-6px;margin-bottom:14px;gap:10px;justify-content:flex-start;flex-wrap:wrap"><form method="post">';
        wp_nonce_field('yb_connect');
        echo '<input type="hidden" name="yb_action" value="connect"/><button type="submit" class="yb-btn yb-btn-filled">' . ($opts['url'] && $opts['secret'] ? 'اتصال مجدد' : 'اتصال خودکار') . '</button></form>';
        if ($connected) {
            echo '<form method="post">';
            wp_nonce_field('yb_disconnect');
            echo '<input type="hidden" name="yb_action" value="disconnect"/><button type="submit" class="yb-danger">قطع اتصال</button></form>';
        }
        echo '</div>';
        echo '<div class="yb-list">';
        $ig = ($opts['ig_username'] || $opts['ig_name'])
            ? trim(($opts['ig_name'] ? $opts['ig_name'] . ' ' : '') . ($opts['ig_username'] ? '@' . ltrim((string)$opts['ig_username'], '@') : ''))
            : '';
        if ($ig) echo '<div class="yb-item"><div><b>اینستاگرام</b><div><span>' . esc_html($ig) . '</span></div></div><span class="yb-pill yb-ok">فعال</span></div>';
        echo '<div class="yb-item"><div><b>آدرس وبهوک</b><div><span class="yb-mask">' . esc_html($maskedUrl) . '</span></div></div></div>';
        echo '<div class="yb-item"><div><b>کلید امنیتی</b><div><span class="yb-mask">' . esc_html($maskedSecret) . '</span></div></div></div>';
        echo '<div class="yb-item"><div><b>شناسه اکانت</b><div><span class="yb-mask">' . esc_html($maskedAccount) . '</span></div></div></div>';
        echo '</div>';
        echo '<div class="yb-actions"><div class="yb-mini">برای تغییر اکانت، از داخل یکتابات دوباره اتصال را انجام دهید</div></div>';
        echo '</div>';

        $paymentUrl = home_url('/?yektabot_pay=1');
        $gwDefault = (string)($opts['payment_gateway_default'] ?? '');
        $paymentEnabled = !empty($opts['payment_enabled']);
        $zarinpalMerchant = (string)($opts['zarinpal_merchant_id'] ?? '');
        $zibalMerchant = (string)($opts['zibal_merchant_id'] ?? '');
        if (!in_array($gwDefault, ['zarinpal', 'zibal'], true)) $gwDefault = '';
        if ($gwDefault === 'zarinpal' && !$zarinpalMerchant) $gwDefault = '';
        if ($gwDefault === 'zibal' && !$zibalMerchant) $gwDefault = '';
        $gwSelect = '<select name="payment_gateway_default" style="height:38px;border-radius:12px;border:1px solid var(--yb-neutral-4);padding:0 34px 0 12px;font-weight:900;background-color:#fff;appearance:none;-webkit-appearance:none;-moz-appearance:none;background-image:url(\'data:image/svg+xml;utf8,<svg xmlns=&quot;http://www.w3.org/2000/svg&quot; viewBox=&quot;0 0 20 20&quot; fill=&quot;none&quot;><path d=&quot;M6 8l4 4 4-4&quot; stroke=&quot;%236B7280&quot; stroke-width=&quot;2&quot; stroke-linecap=&quot;round&quot; stroke-linejoin=&quot;round&quot;/></svg>\');background-repeat:no-repeat;background-position:85% 50%;background-size:16px 16px">'
            . '<option value=""' . ($gwDefault === '' ? ' selected' : '') . '>خودکار</option>'
            . ($zarinpalMerchant ? '<option value="zarinpal"' . ($gwDefault === 'zarinpal' ? ' selected' : '') . '>زرین‌پال</option>' : '')
            . ($zibalMerchant ? '<option value="zibal"' . ($gwDefault === 'zibal' ? ' selected' : '') . '>زیبال</option>' : '')
            . '</select>';
        $paymentToggle = self::switchHtml('payment_enabled', $paymentEnabled, 'فعال');
        echo '<div class="yb-card"><div class="yb-card-head"><div><div class="yb-card-title">پرداخت آنلاین</div><div class="yb-card-desc">ساخت لینک پرداخت روی سایت وردپرس و ارسال نتیجه به یکتابات</div></div><div class="yb-badge"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none"><path d="M3 7h18v10H3V7Z" stroke="white" stroke-width="1.6"/><path d="M6 10h6" stroke="white" stroke-width="1.6" stroke-linecap="round"/><path d="M6 14h4" stroke="white" stroke-width="1.6" stroke-linecap="round"/></svg></div></div>';
        echo '<form method="post">';
        wp_nonce_field('yb_payment');
        echo '<input type="hidden" name="yb_action" value="save_payment"/>';
        echo '<div class="yb-list">';
        echo '<div class="yb-item" style="align-items:flex-start"><div><b>آدرس پرداخت</b><div><span class="yb-mask">' . esc_html($paymentUrl) . '</span></div></div></div>';
        echo '<div class="yb-item" style="align-items:flex-start"><div><b>وضعیت پرداخت آنلاین</b><div><span>فعال/غیرفعال کردن کلی</span></div></div><div>' . $paymentToggle . '</div></div>';
        echo '<div class="yb-item" style="align-items:flex-start"><div><b>درگاه پیش‌فرض</b><div><span>اگر در لینک تعیین نشده باشد</span></div></div>' . $gwSelect . '</div>';
        echo '<div class="yb-item" style="align-items:flex-start"><div><b>زرین‌پال</b><div><span>Merchant ID</span></div></div><div style="display:flex;align-items:center;gap:14px"><span class="yb-mask">' . esc_html($zarinpalMerchant ? self::mask($zarinpalMerchant, 4) : '—') . '</span></div></div>';
        echo '<div class="yb-item" style="align-items:flex-start"><div><b>زیبال</b><div><span>Merchant</span></div></div><div style="display:flex;align-items:center;gap:14px"><span class="yb-mask">' . esc_html($zibalMerchant ? self::mask($zibalMerchant, 4) : '—') . '</span></div></div>';
        echo '</div>';
        echo '<div class="yb-actions"><div class="yb-mini">Merchant از تنظیمات ووکامرس به‌صورت خودکار خوانده می‌شود</div><button type="submit" class="yb-btn yb-btn-filled">ذخیره تنظیمات پرداخت</button></div>';
        echo '</form>';
        echo '</div>';

        echo '<div class="yb-card"><div class="yb-card-head"><div><div class="yb-card-title">وضعیت</div><div class="yb-card-desc">وضعیت اتصال و موتور وبهوک</div></div><div class="yb-badge"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none"><path d="M12 2a10 10 0 1 0 10 10" stroke="white" stroke-width="1.6" stroke-linecap="round"/><path d="M12 6v6l4 2" stroke="white" stroke-width="1.6" stroke-linecap="round" stroke-linejoin="round"/></svg></div></div>';
        echo '<div class="yb-list">';
        foreach ($checks as $check) {
            $pill = '<span class="yb-pill ' . ($check['ok'] ? 'yb-ok' : 'yb-warn') . '">' . ($check['ok'] ? 'فعال' : 'نیاز به بررسی') . '</span>';
            echo '<div class="yb-item"><div><b>' . esc_html($check['label']) . '</b><div><span>' . esc_html($check['detail']) . '</span></div></div>' . $pill . '</div>';
        }
        echo '</div>';
        $fs = get_option(self::FULL_SYNC_OPTION, []);
        $fsToken = is_array($fs) ? (string)($fs['token'] ?? '') : '';
        $fsTotal = is_array($fs) ? (int)($fs['total'] ?? 0) : 0;
        $showFullSyncBox = isset($_GET['yb_full_sync_started']) && (string)$_GET['yb_full_sync_started'] === '1';
        echo '<div id="yb-fullsync" class="yb-item" style="display:' . ($showFullSyncBox ? 'flex' : 'none') . ';margin-top:14px;align-items:stretch;flex-direction:column;gap:10px"><div style="display:flex;align-items:center;justify-content:space-between;gap:10px"><div><b id="yb-fullsync-title">همگام‌سازی کامل در حال انجام است</b><div><span id="yb-fullsync-meta">۰ مورد (بروزرسانی: ۰ / حذف: ۰)</span></div></div><span id="yb-fullsync-pct-pill" class="yb-pill yb-warn">0%</span></div><div style="height:10px;border-radius:999px;background:rgba(214,216,224,.6);overflow:hidden"><div id="yb-fullsync-bar" style="height:100%;width:0%;background:linear-gradient(90deg,var(--yb-primary-400),var(--yb-primary-600))"></div></div></div>';
        $canSync = !empty($opts['url']) && !empty($opts['secret']) && !empty($opts['account_id']);
        echo '<div class="yb-actions" style="margin-top:14px"><div class="yb-mini">' . ($canSync ? 'برای همگام‌سازی کامل محصولات وردپرس با یکتابات' : 'برای همگام‌سازی کامل، ابتدا اتصال را انجام دهید') . '</div><form method="post">';
        wp_nonce_field('yb_full_sync');
        $disabled = !$canSync ? 'disabled style="opacity:.5;pointer-events:none"' : '';
        echo '<input type="hidden" name="yb_action" value="full_sync"/><button id="yb-fullsync-btn" type="submit" class="yb-btn" ' . $disabled . '>همگام‌سازی کامل محصولات</button></form></div>';
        $ajaxNonce = function_exists('wp_create_nonce') ? wp_create_nonce('yb_full_sync_ajax') : '';
        echo '<script>(function(){try{var btn=document.getElementById("yb-fullsync-btn");var box=document.getElementById("yb-fullsync");var bar=document.getElementById("yb-fullsync-bar");var pill=document.getElementById("yb-fullsync-pct-pill");var meta=document.getElementById("yb-fullsync-meta");var title=document.getElementById("yb-fullsync-title");var ajax=window.ajaxurl||"";if(!btn||!box||!bar||!pill||!meta||!title||!ajax)return;var nonce="' . esc_js((string)$ajaxNonce) . '";var runningToken="' . esc_js($showFullSyncBox ? $fsToken : '') . '";function pct(p,t){if(!t||t<=0)return 0;var v=Math.round((p*100)/t);if(v<0)v=0;if(v>100)v=100;return v;}function ui(s){var total=parseInt(s.total||0,10)||0;var processed=parseInt(s.processed||0,10)||0;var updated=parseInt(s.updated||0,10)||0;var deleted=parseInt(s.deleted||0,10)||0;var done=!!s.done;var percent=done?100:pct(processed,total);box.style.display="flex";bar.style.width=percent+"%";pill.textContent=percent+"%";pill.className="yb-pill "+(done?"yb-ok":"yb-warn");title.textContent=done?"همگام‌سازی کامل انجام شد":"همگام‌سازی کامل در حال انجام است";meta.textContent=(total>0?(processed+" از "+total):processed+" مورد")+" (بروزرسانی: "+updated+" / حذف: "+deleted+")";btn.textContent=done?"همگام‌سازی کامل محصولات":"در حال همگام‌سازی...";btn.disabled=!done;btn.style.opacity=done?"":".5";btn.style.pointerEvents=done?"":"none";}function err(msg){box.style.display="flex";bar.style.width="0%";pill.textContent="!";pill.className="yb-pill yb-warn";title.textContent="خطا در همگام‌سازی کامل";meta.textContent=msg||"خطای نامشخص";btn.textContent="همگام‌سازی کامل محصولات";btn.disabled=false;btn.style.opacity="";btn.style.pointerEvents="";}function enc(obj){var s=[];for(var k in obj){if(!Object.prototype.hasOwnProperty.call(obj,k))continue;var v=obj[k];s.push(encodeURIComponent(k)+"="+encodeURIComponent(v===undefined||v===null?"":v));}return s.join("&");}function post(data,cb){data._wpnonce=nonce;var xhr=new XMLHttpRequest();xhr.open("POST",ajax,true);xhr.setRequestHeader("Content-Type","application/x-www-form-urlencoded; charset=UTF-8");xhr.onreadystatechange=function(){if(xhr.readyState!==4)return;var text=xhr.responseText||"";var json=null;try{json=JSON.parse(text);}catch(e){}if(xhr.status<200||xhr.status>=300)return cb("http_"+xhr.status,text,json);if(!json)return cb("bad_json",text,null);cb(null,text,json);};xhr.onerror=function(){cb("network","",null);};xhr.send(enc(data));}function step(token){post({action:"yektabot_full_sync_step",token:token},function(e,raw,res){if(e)return err("خطا در دریافت وضعیت همگام‌سازی");if(!res||!res.success||!res.data)return err("خطا در دریافت وضعیت همگام‌سازی");ui(res.data);if(!res.data.done)setTimeout(function(){step(token);},150);});}function start(){ui({processed:0,updated:0,deleted:0,total:' . esc_js((string)($fsTotal ?: 0)) . ',done:false});post({action:"yektabot_full_sync_start"},function(e,raw,res){if(e)return err("خطا در شروع همگام‌سازی");if(!res||!res.success||!res.data)return err("خطا در شروع همگام‌سازی");runningToken=res.data.token||"";ui(res.data);if(runningToken)setTimeout(function(){step(runningToken);},150);});}var form=btn.closest?btn.closest("form"):null;if(form)form.addEventListener("submit",function(e){e.preventDefault();if(btn.disabled)return;start();},false);if(runningToken){box.style.display="flex";setTimeout(function(){step(runningToken);},150);} }catch(e){}})();</script>';
        echo '</div>';
        echo '</div>';

        $logEnabled = !empty($opts['logs_enabled']);
        echo '<div id="yb-logs" class="yb-card yb-logs"><div class="yb-log-actions"><div><div class="yb-card-title">فعالیت‌ها</div><div class="yb-card-desc">آخرین رویدادهای همگام‌سازی و خطاها</div></div><div style="display:flex;align-items:center;gap:10px">';
        echo '<form method="post">';
        wp_nonce_field('yb_toggle_logs');
        echo '<input type="hidden" name="yb_action" value="toggle_logs"/><button type="submit" class="yb-btn">' . ($logEnabled ? 'غیرفعال کردن لاگ' : 'فعال کردن لاگ') . '</button></form>';
        if ($logEnabled) {
            echo '<form method="post">';
            wp_nonce_field('yb_clear_logs');
            echo '<input type="hidden" name="yb_action" value="clear_logs"/><button type="submit" class="yb-btn">حذف همه لاگ‌ها</button></form>';
        }
        echo '</div></div>';
        if ($logs) {
            echo '<table class="yb-table"><thead><tr><th>زمان</th><th>سطح</th><th>پیام</th><th>جزئیات</th></tr></thead><tbody>';
            foreach ($logs as $log) {
                $time = esc_html($log['time'] ?? '');
                $rawLevel = (string)($log['level'] ?? '');
                $level = esc_html($levelMap[$rawLevel] ?? $rawLevel);
                $rawMessage = (string)($log['message'] ?? '');
                $message = esc_html($messageMap[$rawMessage] ?? $rawMessage);
                $context = esc_html(isset($log['context']) ? json_encode($log['context']) : '');
                $pill = '<span class="yb-pill ' . ($rawLevel === 'info' ? 'yb-ok' : 'yb-warn') . '">' . $level . '</span>';
                echo "<tr><td>$time</td><td>$pill</td><td>$message</td><td><span class=\"yb-code\">$context</span></td></tr>";
            }
            echo '</tbody></table>';
        }
        echo '</div>';

        echo '</main></div></div>';
    }

    private static function startConnect(): void
    {
        $state = function_exists('random_bytes') ? bin2hex(random_bytes(16)) : (function_exists('wp_generate_password') ? wp_generate_password(32, false) : md5(uniqid('', true)));
        self::ensureApiCredentials();
        update_option(self::CONNECT_OPTION, ['state' => $state, 'at' => time(), 'site' => home_url('/')], false);
        $returnUrl = admin_url('admin.php?page=' . self::PAGE . '&yb_action=connected');
        $site = home_url('/');
        $url = self::CONNECT_URL . '?' . http_build_query(['return_url' => $returnUrl, 'state' => $state, 'site' => $site]);
        wp_redirect($url);
        exit;
    }

    private static function handleConnectCallback(): void
    {
        $stored = get_option(self::CONNECT_OPTION, []);
        $storedState = is_array($stored) ? ($stored['state'] ?? null) : null;
        $state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : null;
        if (!$storedState || !$state || !hash_equals((string)$storedState, (string)$state)) {
            delete_option(self::CONNECT_OPTION);
            wp_safe_redirect(admin_url('admin.php?page=' . self::PAGE . '&connected=0'));
            exit;
        }
        $payload = [
            'url' => isset($_GET['url']) ? esc_url_raw((string)$_GET['url']) : 'https://app.yektabot.com/woocommerce/webhook',
            'secret' => isset($_GET['secret']) ? sanitize_text_field((string)$_GET['secret']) : '',
            'account_id' => isset($_GET['account_id']) ? sanitize_text_field((string)$_GET['account_id']) : '',
            'ig_username' => isset($_GET['ig_username']) ? sanitize_text_field((string)$_GET['ig_username']) : '',
            'ig_name' => isset($_GET['ig_name']) ? sanitize_text_field((string)$_GET['ig_name']) : '',
        ];
        $old = self::opts();
        $oldTarget = ($old['url'] && $old['secret']) ? self::target($old) : '';
        update_option(self::OPTION, self::sanitize(array_merge($old, $payload)), false);
        delete_option(self::CONNECT_OPTION);
        $new = self::opts();
        $newTarget = ($new['url'] && $new['secret']) ? self::target($new) : '';
        if ($oldTarget && $newTarget && $oldTarget !== $newTarget) {
            self::cleanupByUrl($oldTarget);
            self::log('info', 'reconnected_detached_previous', ['old_url' => $oldTarget, 'new_url' => $newTarget]);
        }
        self::ensureApiCredentials();
        self::pushCredentials();
        self::track('connected', [
            'site' => home_url('/'),
            'account_id' => (int)$payload['account_id'],
            'ig_username' => (string)$payload['ig_username'],
            'ig_name' => (string)$payload['ig_name'],
        ]);
        if ($payload['url'] && $payload['secret']) self::sync(true);
        wp_safe_redirect(admin_url('admin.php?page=' . self::PAGE . '&connected=1'));
        exit;
    }

    private static function ensureApiCredentials(): void
    {
        try {
            self::loadWoo();
            $opts = self::opts();
            $woo = class_exists('\WooCommerce') || function_exists('WC');
            if (!$woo || !function_exists('wp_generate_password')) {
                self::log('warn', 'api_keys_skipped', ['reason' => 'woocommerce_missing']);
                return;
            }
            global $wpdb;
            if (!isset($wpdb) || !property_exists($wpdb, 'prefix') || !method_exists($wpdb, 'insert') || !method_exists($wpdb, 'get_var') || !method_exists($wpdb, 'prepare')) return;
            $table = $wpdb->prefix . 'woocommerce_api_keys';
            if ($wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table)) !== $table) {
                self::log('warn', 'api_keys_skipped', ['reason' => 'table_missing']);
                return;
            }
            $userId = function_exists('get_current_user_id') ? (int)get_current_user_id() : 0;
            if ($userId <= 0) $userId = 1;
            $consumerKey = trim((string)($opts['consumer_key'] ?? ''));
            $consumerSecret = trim((string)($opts['consumer_secret'] ?? ''));
            $apiKeyId = (int)($opts['api_key_id'] ?? 0);
            if ($consumerKey && $consumerSecret) {
                $hashedKey = function_exists('wc_api_hash') ? wc_api_hash($consumerKey) : hash('sha256', $consumerKey);
                if ($apiKeyId > 0) {
                    $exists = (int)$wpdb->get_var($wpdb->prepare("SELECT key_id FROM {$table} WHERE key_id = %d LIMIT 1", $apiKeyId));
                    if ($exists > 0) return;
                }
                $foundId = (int)$wpdb->get_var($wpdb->prepare("SELECT key_id FROM {$table} WHERE consumer_key = %s LIMIT 1", $hashedKey));
                if ($foundId > 0) {
                    if ($apiKeyId !== $foundId) {
                        $opts['api_key_id'] = (string)$foundId;
                        update_option(self::OPTION, $opts, false);
                    }
                    return;
                }
                $truncated = substr($consumerKey, -7);
                $foundId = (int)$wpdb->get_var($wpdb->prepare("SELECT key_id FROM {$table} WHERE truncated_key = %s AND description = %s ORDER BY key_id DESC LIMIT 1", $truncated, 'Yektabot'));
                if ($foundId > 0 && method_exists($wpdb, 'update')) {
                    $wpdb->update($table, [
                        'user_id' => $userId,
                        'description' => 'Yektabot',
                        'permissions' => 'read_write',
                        'consumer_key' => $hashedKey,
                        'consumer_secret' => $consumerSecret,
                        'truncated_key' => $truncated,
                    ], ['key_id' => $foundId]);
                    $opts['api_key_id'] = (string)$foundId;
                    update_option(self::OPTION, $opts, false);
                    self::log('info', 'api_keys_created', ['id' => (string)$foundId, 'restored' => true]);
                    return;
                }
                $ok = (bool)$wpdb->insert($table, [
                    'user_id' => $userId,
                    'description' => 'Yektabot',
                    'permissions' => 'read_write',
                    'consumer_key' => $hashedKey,
                    'consumer_secret' => $consumerSecret,
                    'nonces' => null,
                    'truncated_key' => $truncated,
                    'last_access' => null,
                ]);
                if ($ok) {
                    $opts['api_key_id'] = (string)($wpdb->insert_id ?? '');
                    update_option(self::OPTION, $opts, false);
                    self::log('info', 'api_keys_created', ['id' => $opts['api_key_id'] ?: null, 'restored' => true]);
                    return;
                }
            }
            if (method_exists($wpdb, 'query')) $wpdb->query($wpdb->prepare("DELETE FROM {$table} WHERE description = %s", 'Yektabot'));
            $consumerKey = 'ck_' . wp_generate_password(32, false, false);
            $consumerSecret = 'cs_' . wp_generate_password(40, false, false);
            $hashedKey = function_exists('wc_api_hash') ? wc_api_hash($consumerKey) : hash('sha256', $consumerKey);
            $ok = (bool)$wpdb->insert($table, [
                'user_id' => $userId,
                'description' => 'Yektabot',
                'permissions' => 'read_write',
                'consumer_key' => $hashedKey,
                'consumer_secret' => $consumerSecret,
                'nonces' => null,
                'truncated_key' => substr($consumerKey, -7),
                'last_access' => null,
            ]);
            if (!$ok) {
                self::log('error', 'api_keys_failed', ['error' => (string)($wpdb->last_error ?? '')]);
                return;
            }
            $opts['consumer_key'] = $consumerKey;
            $opts['consumer_secret'] = $consumerSecret;
            $opts['api_key_id'] = (string)($wpdb->insert_id ?? '');
            update_option(self::OPTION, $opts, false);
            self::log('info', 'api_keys_created', ['id' => $opts['api_key_id'] ?: null]);
        } catch (\Throwable $e) {
            self::log('error', 'api_keys_failed', ['error' => $e->getMessage()]);
        }
    }

    private static function pushCredentials(): void
    {
        try {
            $opts = self::opts();
            if (empty($opts['account_id']) || empty($opts['secret']) || empty($opts['consumer_key']) || empty($opts['consumer_secret'])) return;
            if (!function_exists('wp_remote_post') || !function_exists('wp_remote_retrieve_response_code') || !function_exists('wp_remote_retrieve_body')) return;
            $body = wp_json_encode([
                'account_id' => (int)$opts['account_id'],
                'site' => home_url('/'),
                'consumer_key' => (string)$opts['consumer_key'],
                'consumer_secret' => (string)$opts['consumer_secret'],
            ]);
            if (!$body) return;
            $sig = base64_encode(hash_hmac('sha256', $body, (string)$opts['secret'], true));
            $res = wp_remote_post(self::CREDENTIALS_URL, [
                'timeout' => 15,
                'headers' => [
                    'Content-Type' => 'application/json',
                    'X-Yektabot-Signature' => $sig,
                ],
                'body' => $body,
            ]);
            if (is_wp_error($res)) {
                self::log('error', 'api_keys_push_failed', ['error' => $res->get_error_message()]);
                return;
            }
            $code = (int)wp_remote_retrieve_response_code($res);
            self::log(($code >= 200 && $code < 300) ? 'info' : 'warn', 'api_keys_pushed', ['status' => $code, 'body_len' => strlen((string)wp_remote_retrieve_body($res))]);
        } catch (\Throwable $e) {
            self::log('error', 'api_keys_push_failed', ['error' => $e->getMessage()]);
        }
    }

    private static function opts(): array
    {
        $defaults = ['url' => '', 'secret' => '', 'account_id' => '', 'ig_username' => '', 'ig_name' => '', 'consumer_key' => '', 'consumer_secret' => '', 'api_key_id' => '', 'logs_enabled' => '1', 'payment_enabled' => '1', 'payment_gateway_default' => '', 'zarinpal_merchant_id' => '', 'zibal_merchant_id' => '', 'payment_merchant_source' => 'auto'];
        $value = get_option(self::OPTION, []);
        return array_merge($defaults, is_array($value) ? $value : []);
    }

    private static function target(array $opts): string
    {
        $base = (string)($opts['url'] ?? '');
        if (!$opts['account_id']) return $base;
        $parts = parse_url($base);
        if (!$parts) return $base;
        $query = [];
        if (!empty($parts['query'])) parse_str($parts['query'], $query);
        unset($query['account_id']);
        $query['account_id'] = (string)$opts['account_id'];
        $scheme = isset($parts['scheme']) ? ($parts['scheme'] . '://') : '';
        $user = $parts['user'] ?? '';
        $pass = isset($parts['pass']) ? ':' . $parts['pass'] : '';
        $auth = $user ? $user . $pass . '@' : '';
        $host = $parts['host'] ?? '';
        $port = isset($parts['port']) ? ':' . $parts['port'] : '';
        $path = $parts['path'] ?? '';
        $fragment = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';
        $rebuilt = $scheme . $auth . $host . $port . $path;
        $rebuilt .= $query ? ('?' . http_build_query($query)) : '';
        return $rebuilt . $fragment;
    }

    public static function sanitize($value): array
    {
        $value = is_array($value) ? $value : [];
        $gateway = sanitize_text_field($value['payment_gateway_default'] ?? '');
        if (!in_array($gateway, ['zarinpal', 'zibal'], true)) $gateway = '';
        $source = sanitize_text_field($value['payment_merchant_source'] ?? 'auto');
        if (!in_array($source, ['auto'], true)) $source = 'auto';
        return [
            'url' => esc_url_raw($value['url'] ?? ''),
            'secret' => sanitize_text_field($value['secret'] ?? ''),
            'account_id' => sanitize_text_field($value['account_id'] ?? ''),
            'ig_username' => sanitize_text_field($value['ig_username'] ?? ''),
            'ig_name' => sanitize_text_field($value['ig_name'] ?? ''),
            'consumer_key' => sanitize_text_field($value['consumer_key'] ?? ''),
            'consumer_secret' => sanitize_text_field($value['consumer_secret'] ?? ''),
            'api_key_id' => sanitize_text_field($value['api_key_id'] ?? ''),
            'logs_enabled' => !empty($value['logs_enabled']) ? '1' : '0',
            'payment_enabled' => !empty($value['payment_enabled']) ? '1' : '0',
            'payment_gateway_default' => $gateway,
            'zarinpal_merchant_id' => sanitize_text_field($value['zarinpal_merchant_id'] ?? ''),
            'zibal_merchant_id' => sanitize_text_field($value['zibal_merchant_id'] ?? ''),
            'payment_merchant_source' => $source,
        ];
    }

    private static function toggleLogs(): void
    {
        $opts = self::opts();
        $opts['logs_enabled'] = empty($opts['logs_enabled']) ? '1' : '0';
        update_option(self::OPTION, $opts, false);
        if (empty($opts['logs_enabled'])) self::clearLogs();
        wp_safe_redirect(admin_url('admin.php?page=' . self::PAGE . '#yb-logs'));
        exit;
    }

    private static function startFullSync(): void
    {
        $opts = self::opts();
        if (empty($opts['url']) || empty($opts['secret']) || empty($opts['account_id'])) {
            wp_safe_redirect(admin_url('admin.php?page=' . self::PAGE . '&full_sync=0'));
            exit;
        }
        $token = function_exists('random_bytes') ? bin2hex(random_bytes(16)) : (function_exists('wp_generate_password') ? wp_generate_password(32, false) : md5(uniqid('', true)));
        update_option(self::FULL_SYNC_OPTION, [
            'token' => $token,
            'last_id' => 0,
            'processed' => 0,
            'updated' => 0,
            'deleted' => 0,
            'total' => self::fullSyncTotal(),
            'started_at' => time(),
        ], false);
        self::log('info', 'full_sync_started', []);
        wp_safe_redirect(admin_url('admin.php?page=' . self::PAGE . '&yb_full_sync_started=1'));
        exit;
    }

    public static function ajaxFullSyncStart(): void
    {
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'forbidden'], 403);
        check_ajax_referer('yb_full_sync_ajax');
        $opts = self::opts();
        if (empty($opts['url']) || empty($opts['secret']) || empty($opts['account_id'])) wp_send_json_error(['message' => 'not_ready'], 422);
        $token = function_exists('random_bytes') ? bin2hex(random_bytes(16)) : (function_exists('wp_generate_password') ? wp_generate_password(32, false) : md5(uniqid('', true)));
        $state = [
            'token' => $token,
            'last_id' => 0,
            'processed' => 0,
            'updated' => 0,
            'deleted' => 0,
            'total' => self::fullSyncTotal(),
            'started_at' => time(),
        ];
        update_option(self::FULL_SYNC_OPTION, $state, false);
        self::log('info', 'full_sync_started', []);
        wp_send_json_success($state);
    }

    public static function ajaxFullSyncStep(): void
    {
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'forbidden'], 403);
        check_ajax_referer('yb_full_sync_ajax');
        $token = isset($_POST['token']) ? sanitize_text_field((string)$_POST['token']) : '';
        $state = get_option(self::FULL_SYNC_OPTION, []);
        $stored = is_array($state) ? (string)($state['token'] ?? '') : '';
        if (!$stored || !$token || !hash_equals($stored, $token)) wp_send_json_error(['message' => 'invalid_state'], 422);
        $total = (int)($state['total'] ?? 0);
        if ($total <= 0) {
            $total = self::fullSyncTotal();
            $state['total'] = $total;
        }
        $afterId = (int)($state['last_id'] ?? 0);
        $batch = self::fullSyncBatch($afterId, 60);
        if (!$batch) {
            delete_option(self::FULL_SYNC_OPTION);
            $processed = (int)($state['processed'] ?? 0);
            $updated = (int)($state['updated'] ?? 0);
            $deleted = (int)($state['deleted'] ?? 0);
            self::log('info', 'full_sync_completed', ['processed' => $processed, 'updated' => $updated, 'deleted' => $deleted]);
            wp_send_json_success(['done' => true, 'token' => $token, 'processed' => $processed, 'updated' => $updated, 'deleted' => $deleted, 'total' => $total]);
        }
        foreach ($batch as $row) {
            $id = (int)($row['id'] ?? 0);
            if ($id <= 0) continue;
            $status = (string)($row['status'] ?? '');
            $topic = $status === 'trash' ? 'product.deleted' : 'product.updated';
            self::directDeliverPayload($topic, ['id' => $id, 'post_status' => $status], false);
            $state['processed'] = (int)($state['processed'] ?? 0) + 1;
            $state[$status === 'trash' ? 'deleted' : 'updated'] = (int)($state[$status === 'trash' ? 'deleted' : 'updated'] ?? 0) + 1;
            $state['last_id'] = $id;
        }
        update_option(self::FULL_SYNC_OPTION, $state, false);
        wp_send_json_success([
            'done' => false,
            'token' => $token,
            'processed' => (int)($state['processed'] ?? 0),
            'updated' => (int)($state['updated'] ?? 0),
            'deleted' => (int)($state['deleted'] ?? 0),
            'total' => (int)($state['total'] ?? 0),
        ]);
    }

    private static function continueFullSync(): ?array
    {
        $state = get_option(self::FULL_SYNC_OPTION, []);
        $token = isset($_GET['token']) ? sanitize_text_field((string)$_GET['token']) : '';
        $stored = is_array($state) ? (string)($state['token'] ?? '') : '';
        if (!$stored || !$token || !hash_equals($stored, $token)) {
            delete_option(self::FULL_SYNC_OPTION);
            wp_safe_redirect(admin_url('admin.php?page=' . self::PAGE . '&full_sync=0'));
            exit;
        }
        $total = (int)($state['total'] ?? 0);
        if ($total <= 0) {
            $total = self::fullSyncTotal();
            $state['total'] = $total;
        }
        $afterId = (int)($state['last_id'] ?? 0);
        $batch = self::fullSyncBatch($afterId, 60);
        if (!$batch) {
            delete_option(self::FULL_SYNC_OPTION);
            $processed = (int)($state['processed'] ?? 0);
            $updated = (int)($state['updated'] ?? 0);
            $deleted = (int)($state['deleted'] ?? 0);
            self::log('info', 'full_sync_completed', ['processed' => $processed, 'updated' => $updated, 'deleted' => $deleted]);
            return ['done' => true, 'token' => $token, 'processed' => $processed, 'updated' => $updated, 'deleted' => $deleted, 'total' => $total];
        }
        foreach ($batch as $row) {
            $id = (int)($row['id'] ?? 0);
            if ($id <= 0) continue;
            $status = (string)($row['status'] ?? '');
            $topic = $status === 'trash' ? 'product.deleted' : 'product.updated';
            self::directDeliverPayload($topic, ['id' => $id, 'post_status' => $status], false);
            $state['processed'] = (int)($state['processed'] ?? 0) + 1;
            $state[$status === 'trash' ? 'deleted' : 'updated'] = (int)($state[$status === 'trash' ? 'deleted' : 'updated'] ?? 0) + 1;
            $state['last_id'] = $id;
        }
        update_option(self::FULL_SYNC_OPTION, $state, false);
        $processed = (int)($state['processed'] ?? 0);
        $updated = (int)($state['updated'] ?? 0);
        $deleted = (int)($state['deleted'] ?? 0);
        return ['done' => false, 'token' => $token, 'processed' => $processed, 'updated' => $updated, 'deleted' => $deleted, 'total' => $total];
    }

    private static function fullSyncTotal(): int
    {
        global $wpdb;
        if (!isset($wpdb) || !method_exists($wpdb, 'get_var') || !isset($wpdb->posts)) return 0;
        $statuses = ['publish', 'draft', 'pending', 'private', 'future', 'trash'];
        $placeholders = implode(',', array_fill(0, count($statuses), '%s'));
        $sql = "SELECT COUNT(1) FROM {$wpdb->posts} WHERE post_type = %s AND post_status IN ({$placeholders})";
        $args = array_merge(['product'], $statuses);
        $count = (int)$wpdb->get_var($wpdb->prepare($sql, $args));
        return max(0, $count);
    }

    private static function fullSyncBatch(int $afterId, int $limit): array
    {
        global $wpdb;
        if (!isset($wpdb) || !method_exists($wpdb, 'get_results')) return [];
        $limit = max(1, min(200, $limit));
        $afterId = max(0, $afterId);
        $statuses = ['publish', 'draft', 'pending', 'private', 'future', 'trash'];
        $placeholders = implode(',', array_fill(0, count($statuses), '%s'));
        $sql = "SELECT ID, post_status FROM {$wpdb->posts} WHERE post_type = %s AND post_status IN ({$placeholders}) AND ID > %d ORDER BY ID ASC LIMIT %d";
        $args = array_merge(['product'], $statuses, [$afterId, $limit]);
        $rows = $wpdb->get_results($wpdb->prepare($sql, $args), ARRAY_A);
        if (!is_array($rows)) return [];
        return array_map(fn($r) => ['id' => (int)($r['ID'] ?? 0), 'status' => (string)($r['post_status'] ?? '')], $rows);
    }

    private static function directDeliverPayload(string $topic, array $payload, bool $blocking = true): void
    {
        $opts = self::opts();
        if (empty($opts['url']) || empty($opts['secret'])) {
            self::log('warn', 'direct_delivery_skipped', ['reason' => 'not_connected', 'topic' => $topic]);
            return;
        }
        if (!function_exists('wp_remote_post')) return;
        $url = self::target($opts);
        $body = wp_json_encode($payload);
        if (!$body) return;
        $sig = base64_encode(hash_hmac('sha256', $body, (string)$opts['secret'], true));
        $headers = [
            'Content-Type' => 'application/json',
            'X-WC-Webhook-Topic' => $topic,
            'X-WC-Webhook-Signature' => $sig,
            'X-WC-Webhook-Source' => home_url('/'),
        ];
        $args = [
            'timeout' => $blocking ? 15 : 0.01,
            'blocking' => $blocking,
            'headers' => $headers,
            'body' => $body,
            'redirection' => 0,
        ];
        $resp = wp_remote_post($url, $args);
        if ($blocking) {
            if (is_wp_error($resp)) {
                self::log('error', 'direct_delivery', ['topic' => $topic, 'error' => $resp->get_error_message()]);
                return;
            }
            $code = function_exists('wp_remote_retrieve_response_code') ? (int)wp_remote_retrieve_response_code($resp) : null;
            self::log(($code >= 200 && $code < 300) ? 'info' : 'warn', 'direct_delivery', ['topic' => $topic, 'status' => $code, 'url' => $url]);
        }
    }

    private static function cleanupByUrl(string $url): void
    {
        self::loadWoo();
        $table = self::webhookTable();
        if ($table) {
            global $wpdb;
            $like = $wpdb->esc_like('Yektabot ') . '%';
            $wpdb->query($wpdb->prepare("DELETE FROM {$table} WHERE name LIKE %s AND delivery_url = %s", $like, $url));
        }
        if (class_exists('\WC_Webhooks') && method_exists('\WC_Webhooks', 'get_webhooks')) {
            foreach (self::webhooks() as $hook) {
                if ($hook->get_name() === 'Yektabot ' . $hook->get_topic() && $hook->get_delivery_url() === $url) $hook->delete(true);
            }
        } else {
            $ids = get_posts(['post_type' => 'shop_webhook', 'post_status' => 'any', 'posts_per_page' => -1, 'fields' => 'ids']);
            foreach ($ids as $id) {
                $title = (string)get_the_title((int)$id);
                if (!str_starts_with($title, 'Yektabot ')) continue;
                $delivery = (string)(get_post_meta((int)$id, '_webhook_delivery_url', true) ?: get_post_meta((int)$id, 'webhook_delivery_url', true));
                if ($delivery !== $url) continue;
                wp_delete_post((int)$id, true);
            }
        }
        delete_option(self::OPTION . '_synced');
    }

    private static function loadWoo(): void
    {
        if (!function_exists('WC')) return;
        try { WC(); } catch (\Throwable $e) {}
    }

    private static function webhookTable(): ?string
    {
        if (self::$webhookTable !== null) return self::$webhookTable ?: null;
        global $wpdb;
        if (!isset($wpdb) || !isset($wpdb->prefix)) return self::$webhookTable = '';
        foreach ([$wpdb->prefix . 'wc_webhooks', $wpdb->prefix . 'woocommerce_webhooks'] as $table) {
            $found = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table));
            if ($found === $table) return self::$webhookTable = $table;
        }
        return self::$webhookTable = '';
    }

    private static function webhooks(): array
    {
        return method_exists('\WC_Webhooks', 'get_webhooks') ? \WC_Webhooks::get_webhooks() : [];
    }

    private static function log(string $level, string $message, array $context): void
    {
        if (empty(self::opts()['logs_enabled'])) return;
        $logs = self::logs();
        $logs[] = ['time' => current_time('mysql'), 'level' => $level, 'message' => $message, 'context' => $context];
        if (count($logs) > 50) $logs = array_slice($logs, -50);
        update_option(self::LOG_OPTION, $logs, false);
    }

    private static function logs(): array
    {
        $value = get_option(self::LOG_OPTION, []);
        return is_array($value) ? $value : [];
    }

    public static function sync(bool $validated = false): void
    {
        if (!$validated && !self::ready(true)) return;
        $opts = self::opts();
        foreach (self::TOPICS as $topic) self::ensure($topic, $opts);
        update_option(self::OPTION . '_synced', (string)self::signature());
        self::log('info', 'synced', ['topics' => self::TOPICS, 'url' => self::target($opts)]);
    }

    private static function ready(bool $log = false): bool
    {
        self::loadWoo();
        $opts = self::opts();
        if (!$opts['url'] || !$opts['secret']) {
            if ($log) self::log('warn', 'config_missing', $opts);
            return false;
        }
        $woo = class_exists('\WooCommerce') || function_exists('WC');
        if (!$woo) {
            if ($log) self::log('error', 'woocommerce_inactive', []);
            return false;
        }
        if (class_exists('\WC_Webhook') && class_exists('\WC_Webhooks')) return true;
        $table = self::webhookTable();
        if ($table) {
            if ($log) self::log('warn', 'fallback_db', ['wc_version' => defined('WC_VERSION') ? WC_VERSION : null, 'table' => $table]);
            return true;
        }
        if ($log) self::log('warn', 'fallback_cpt', ['wc_version' => defined('WC_VERSION') ? WC_VERSION : null]);
        return true;
    }

    private static function ensure(string $topic, array $opts): void
    {
        self::loadWoo();
        try {
            $url = self::target($opts);
            if (!class_exists('\WC_Webhook') || !class_exists('\WC_Webhooks')) {
                $table = self::webhookTable();
                if ($table) {
                    self::ensureTable($table, $topic, $url, $opts['secret']);
                    return;
                }
                self::ensureCpt($topic, $url, $opts['secret']);
                return;
            }
            $name = 'Yektabot ' . $topic;
            $matches = [];
            foreach (self::webhooks() as $hook) {
                if ($hook->get_topic() !== $topic) continue;
                $hookName = $hook->get_name();
                if (!$hookName || !str_starts_with($hookName, 'Yektabot ')) continue;
                $matches[] = $hook;
            }
            $existing = null;
            foreach ($matches as $hook) {
                if ($hook->get_name() === $name) { $existing = $hook; break; }
            }
            if (!$existing && $matches) $existing = $matches[0];
            if ($existing) {
                foreach ($matches as $hook) if ($hook !== $existing) $hook->delete(true);
                $existing->set_name($name);
                $existing->set_delivery_url($url);
                $existing->set_secret($opts['secret']);
                $existing->set_status('active');
                $existing->save();
                self::log('info', 'webhook_updated', ['topic' => $topic, 'url' => $url]);
                return;
            }
            $hook = new \WC_Webhook();
            $hook->set_name($name);
            $hook->set_topic($topic);
            $hook->set_delivery_url($url);
            $hook->set_secret($opts['secret']);
            $hook->set_status('active');
            $hook->save();
            self::log('info', 'webhook_created', ['topic' => $topic, 'url' => $url]);
        } catch (\Throwable $e) {
            self::log('error', 'webhook_failed', ['topic' => $topic, 'error' => $e->getMessage()]);
        }
    }

    private static function ensureTable(string $table, string $topic, string $url, string $secret): void
    {
        global $wpdb;
        $now = current_time('mysql');
        $nowGmt = current_time('mysql', 1);
        $name = 'Yektabot ' . $topic;
        $userId = get_current_user_id() ?: 0;
        $ids = $wpdb->get_col($wpdb->prepare("SELECT webhook_id FROM {$table} WHERE topic = %s AND name = %s ORDER BY webhook_id DESC", $topic, $name));
        if (empty($ids)) {
            $like = method_exists($wpdb, 'esc_like') ? ($wpdb->esc_like('Yektabot ') . '%') : 'Yektabot %';
            $ids = $wpdb->get_col($wpdb->prepare("SELECT webhook_id FROM {$table} WHERE topic = %s AND name LIKE %s ORDER BY webhook_id DESC", $topic, $like));
        }
        $id = !empty($ids) ? (int)$ids[0] : 0;
        if ($id > 0) {
            $wpdb->update($table, [
                'status' => 'active',
                'name' => $name,
                'user_id' => $userId,
                'delivery_url' => $url,
                'secret' => $secret,
                'topic' => $topic,
                'date_modified' => $now,
                'date_modified_gmt' => $nowGmt,
                'api_version' => 3,
                // must be 0 when idle; WooCommerce sets it to 1 only while delivering.
                'pending_delivery' => 0,
            ], ['webhook_id' => $id]);
            foreach (array_slice($ids, 1) as $dupId) {
                $dupId = (int)$dupId;
                if ($dupId > 0) $wpdb->delete($table, ['webhook_id' => $dupId], ['%d']);
            }
            self::log('info', 'webhook_updated', ['topic' => $topic, 'url' => $url, 'engine' => 'table', 'id' => $id]);
            return;
        }
        $wpdb->insert($table, [
            'status' => 'active',
            'name' => $name,
            'user_id' => $userId,
            'delivery_url' => $url,
            'secret' => $secret,
            'topic' => $topic,
            'date_created' => $now,
            'date_created_gmt' => $nowGmt,
            'date_modified' => $now,
            'date_modified_gmt' => $nowGmt,
            'api_version' => 3,
            'failure_count' => 0,
            'pending_delivery' => 0,
        ]);
        self::log('info', 'webhook_created', ['topic' => $topic, 'url' => $url, 'engine' => 'table', 'id' => (int)($wpdb->insert_id ?? 0)]);
    }

    private static function ensureCpt(string $topic, string $url, string $secret): void
    {
        $ids = self::findYektabotWebhookPosts($topic);
        $id = !empty($ids) ? (int)$ids[0] : 0;
        if ($id > 0) {
            self::updateWebhookPost($id, $topic, $url, $secret);
            foreach (array_slice($ids, 1) as $dupId) wp_delete_post((int)$dupId, true);
            self::log('info', 'webhook_updated', ['topic' => $topic, 'url' => $url, 'engine' => 'cpt', 'id' => $id]);
            return;
        }
        $id = wp_insert_post([
            'post_type' => 'shop_webhook',
            'post_status' => 'publish',
            'post_title' => 'Yektabot ' . $topic,
            'post_author' => get_current_user_id() ?: 0,
        ], true);
        if (is_wp_error($id)) {
            self::log('error', 'webhook_failed', ['topic' => $topic, 'engine' => 'cpt', 'error' => $id->get_error_message()]);
            return;
        }
        $id = (int)$id;
        if ($id <= 0) {
            self::log('error', 'webhook_failed', ['topic' => $topic, 'engine' => 'cpt', 'error' => 'invalid_id']);
            return;
        }
        self::updateWebhookPost($id, $topic, $url, $secret);
        self::log('info', 'webhook_created', ['topic' => $topic, 'url' => $url, 'engine' => 'cpt', 'id' => $id]);
    }

    private static function findYektabotWebhookPosts(string $topic): array
    {
        $title = 'Yektabot ' . $topic;
        $ids = get_posts([
            'post_type' => 'shop_webhook',
            'post_status' => 'any',
            'posts_per_page' => 50,
            'fields' => 'ids',
            'orderby' => 'ID',
            'order' => 'DESC',
            'meta_query' => [['key' => '_webhook_topic', 'value' => $topic, 'compare' => '=']],
        ]);
        $ids = array_values(array_filter($ids, fn($id) => (($t = (string)get_the_title((int)$id)) === $title) || str_starts_with($t, 'Yektabot ')));
        if (!empty($ids)) return $ids;
        $ids = get_posts([
            'post_type' => 'shop_webhook',
            'post_status' => 'any',
            'posts_per_page' => 50,
            'fields' => 'ids',
            'orderby' => 'ID',
            'order' => 'DESC',
            'meta_query' => [['key' => 'webhook_topic', 'value' => $topic, 'compare' => '=']],
        ]);
        return array_values(array_filter($ids, fn($id) => (($t = (string)get_the_title((int)$id)) === $title) || str_starts_with($t, 'Yektabot ')));
    }

    private static function updateWebhookPost(int $id, string $topic, string $url, string $secret): void
    {
        wp_update_post(['ID' => $id, 'post_title' => 'Yektabot ' . $topic, 'post_status' => 'publish']);
        $userId = get_current_user_id() ?: 0;
        $pairs = [
            '_webhook_topic' => $topic,
            '_webhook_delivery_url' => $url,
            '_webhook_secret' => $secret,
            '_webhook_status' => 'active',
            '_webhook_user_id' => $userId,
            '_webhook_api_version' => 3,
        ];
        foreach ($pairs as $key => $value) update_post_meta($id, $key, $value);
        $pairs = [
            'webhook_topic' => $topic,
            'webhook_delivery_url' => $url,
            'webhook_secret' => $secret,
            'webhook_status' => 'active',
        ];
        foreach ($pairs as $key => $value) update_post_meta($id, $key, $value);
    }

    private static function signature(): string
    {
        $opts = self::opts();
        return md5(json_encode($opts));
    }

    private static function clearLogs(): void
    {
        update_option(self::LOG_OPTION, [], false);
    }

    private static function renderUpdateRequired(): void
    {
        $meta = self::$updateMeta;
        $min = esc_html($meta['min'] ?? '');
        $msg = esc_html($meta['message'] ?? 'نسخه افزونه قدیمی است؛ لطفاً به‌روزرسانی کنید.');
        $download = esc_url($meta['download'] ?? 'https://downloads.wordpress.org/plugin/yektabot.latest-stable.zip');
        echo '<div class="wrap" dir="rtl" style="font-family:IRANSans,system-ui;-webkit-font-smoothing:antialiased">';
        echo '<div style="max-width:820px;margin:40px auto;padding:24px 28px;border:1px solid #e2e8f0;border-radius:16px;background:#fff;box-shadow:0 12px 30px rgba(0,0,0,.06)">';
        echo '<h2 style="margin-top:0;font-weight:800;color:#111827;">به‌روزرسانی ضروری افزونه یکتابات</h2>';
        echo '<p style="color:#374151;font-weight:600;margin:12px 0;">' . $msg . '</p>';
        if ($min) echo '<p style="color:#6b7280;font-weight:600;margin:6px 0;">حداقل نسخه مورد نیاز: ' . $min . ' | نسخه فعلی: ' . esc_html(self::VERSION) . '</p>';
        echo '<div style="margin-top:18px;display:flex;gap:12px;align-items:center;">';
        echo '<a href="' . $download . '" class="button button-primary" style="height:42px;display:inline-flex;align-items:center;padding:0 16px;border-radius:8px;font-weight:700;">دانلود آخرین نسخه</a>';
        echo '<a href="update-core.php" class="button" style="height:42px;display:inline-flex;align-items:center;padding:0 14px;border-radius:8px;font-weight:700;">برو به بخش به‌روزرسانی‌ها</a>';
        echo '</div>';
        echo '</div>';
        echo '</div>';
    }

    private static function readyDetails(): array
    {
        self::loadWoo();
        $opts = self::opts();
        $checks = [];
        $woo = class_exists('\WooCommerce') || function_exists('WC');
        $version = defined('WC_VERSION') ? WC_VERSION : null;
        $classes = class_exists('\WC_Webhook') && class_exists('\WC_Webhooks');
        $table = self::webhookTable();
        $registered = function_exists('post_type_exists') ? post_type_exists('shop_webhook') : null;
        $hasAny = !empty(get_posts(['post_type' => 'shop_webhook', 'post_status' => 'any', 'posts_per_page' => 1, 'fields' => 'ids']));
        $engine = $woo ? ($classes ? 'classes' : ($table ? 'table' : 'cpt')) : 'none';
        $checks[] = ['label' => 'ووکامرس', 'ok' => $woo, 'detail' => $woo ? ('فعال' . ($version ? " (نسخه $version)" : '')) : 'پیدا نشد یا فعال نیست'];
        $checks[] = ['label' => 'موتور وبهوک', 'ok' => $engine !== 'none', 'detail' => $engine === 'classes' ? 'کلاس‌های ووکامرس' : ($engine === 'table' ? 'جدول دیتابیس (wc_webhooks)' : ($engine === 'cpt' ? 'پست‌تایپ (shop_webhook)' : 'در دسترس نیست'))];
        $checks[] = ['label' => 'جدول wc_webhooks', 'ok' => (bool)$table, 'detail' => $table ?: 'وجود ندارد'];
        $checks[] = ['label' => 'آدرس وبهوک', 'ok' => (bool)$opts['url'], 'detail' => $opts['url'] ?: 'ثبت نشده'];
        $checks[] = ['label' => 'کلید امنیتی', 'ok' => (bool)$opts['secret'], 'detail' => $opts['secret'] ? 'تنظیم شده' : 'ثبت نشده'];
        return $checks;
    }

    private static function installId(): string
    {
        $state = get_option(self::TRACK_OPTION, []);
        $state = is_array($state) ? $state : [];
        if (empty($state['id'])) {
            $state['id'] = function_exists('random_bytes') ? bin2hex(random_bytes(16)) : (function_exists('wp_generate_password') ? wp_generate_password(32, false) : md5(uniqid('', true)));
            $state['created_at'] = time();
            update_option(self::TRACK_OPTION, $state, false);
        }
        update_option(self::INSTALL_ID_OPTION, $state['id'], false);
        return (string)$state['id'];
    }

    private static function fontUrl(): ?string
    {
        if (!function_exists('wp_upload_dir') || !function_exists('wp_safe_remote_get') || !function_exists('wp_remote_retrieve_response_code') || !function_exists('wp_remote_retrieve_body')) return null;
        $upload = wp_upload_dir(null, false);
        if (!is_array($upload) || !empty($upload['error'])) return null;
        $baseDir = rtrim((string)($upload['basedir'] ?? ''), '/');
        $baseUrl = rtrim((string)($upload['baseurl'] ?? ''), '/');
        if (!$baseDir || !$baseUrl) return null;
        $dir = $baseDir . '/' . self::FONT_CACHE_DIR;
        $urlDir = $baseUrl . '/' . self::FONT_CACHE_DIR;
        $path = $dir . '/' . self::FONT_CACHE_FILE;
        $url = $urlDir . '/' . self::FONT_CACHE_FILE;
        if (is_file($path) && (int)@filesize($path) > 1024) return $url;
        if (!function_exists('wp_mkdir_p') || !wp_mkdir_p($dir)) return null;
        $res = wp_safe_remote_get(self::FONT_REMOTE_URL, ['timeout' => 20, 'redirection' => 2]);
        if (is_wp_error($res)) return null;
        $code = (int)wp_remote_retrieve_response_code($res);
        $body = (string)wp_remote_retrieve_body($res);
        $len = strlen($body);
        if ($code !== 200 || $len < 1024 || $len > 5000000) return null;
        if (@file_put_contents($path, $body, LOCK_EX) === false) return null;
        return $url;
    }

    private static function mask(string $value, int $visible): string
    {
        $value = trim($value);
        $len = strlen($value);
        if ($len <= $visible) return str_repeat('•', max(0, $len - 1)) . substr($value, -1);
        return str_repeat('•', max(8, $len - $visible)) . substr($value, -$visible);
    }

    private static function maskUrl(string $url, string $accountId): string
    {
        $p = parse_url($url);
        $host = $p['host'] ?? 'app.yektabot.com';
        $path = $p['path'] ?? '/woocommerce/webhook';
        $shortPath = '/…' . (str_ends_with($path, '/webhook') ? '/webhook' : '');
        $suffix = $accountId ? ('?account_id=' . self::mask($accountId, 2)) : '';
        return $host . $shortPath . $suffix;
    }

    private static function savePaymentSettings(): void
    {
        $opts = self::opts();
        $payload = [
            'payment_gateway_default' => isset($_POST['payment_gateway_default']) ? sanitize_text_field((string)$_POST['payment_gateway_default']) : '',
            'payment_enabled' => !empty($_POST['payment_enabled']) ? '1' : '0',
            'payment_merchant_source' => 'auto',
        ];
        update_option(self::OPTION, self::sanitize(array_merge($opts, $payload)), false);
        self::syncPaymentSettingsFromWoo(true);
        self::track('heartbeat', []);
        wp_safe_redirect(admin_url('admin.php?page=' . self::PAGE . '&payment=1'));
        exit;
    }

    public static function maybeSyncPaymentSettings(): void
    {
        if (!is_admin() || !current_user_can('manage_options')) return;
        $state = get_option(self::PAYMENT_SYNC_OPTION, []);
        $state = is_array($state) ? $state : [];
        $last = (int)($state['last'] ?? 0);
        if ($last > 0 && (time() - $last) < 21600) return;
        self::syncPaymentSettingsFromWoo(false);
    }

    private static function syncPaymentSettingsFromWoo(bool $force): void
    {
        try {
            self::loadWoo();
            $opts = self::opts();
            $detected = ['zarinpal_merchant_id' => '', 'zibal_merchant_id' => ''];
            $defaultGateway = function_exists('get_option') ? (string)get_option('woocommerce_default_gateway', '') : '';
            if (class_exists('\WC') && function_exists('WC')) {
                try {
                    $gws = WC()->payment_gateways && method_exists(WC()->payment_gateways, 'payment_gateways')
                        ? WC()->payment_gateways->payment_gateways()
                        : [];
                    if (is_array($gws)) {
                        foreach ($gws as $id => $gw) {
                            $id = (string)$id;
                            $probe = strtolower($id . ' ' . (is_object($gw) && property_exists($gw, 'method_title') ? (string)$gw->method_title : '') . ' ' . (is_object($gw) && property_exists($gw, 'title') ? (string)$gw->title : ''));
                            $kind = str_contains($probe, 'zarin') ? 'zarinpal' : (str_contains($probe, 'zibal') ? 'zibal' : null);
                            if (!$kind) continue;
                            $s = get_option('woocommerce_' . $id . '_settings', []);
                            if (!is_array($s) || (($s['enabled'] ?? 'yes') !== 'yes')) continue;
                            $keys = $kind === 'zarinpal'
                                ? ['merchantcode', 'merchant_id', 'merchantId', 'MerchantID', 'merchant', 'merchantCode']
                                : ['merchant', 'merchant_id', 'merchantId', 'MerchantID', 'merchantcode', 'merchantCode'];
                            foreach ($keys as $k) {
                                $v = isset($s[$k]) ? trim((string)$s[$k]) : '';
                                if (!$v) continue;
                                $detected[$kind === 'zarinpal' ? 'zarinpal_merchant_id' : 'zibal_merchant_id'] = $v;
                                break;
                            }
                        }
                    }
                } catch (\Throwable $e) {
                }
            }
            $map = [
                'zarinpal' => [
                    'options' => ['woocommerce_zarinpal_settings', 'woocommerce_wc_zarinpal_settings', 'woocommerce_persian_gateway_zarinpal_settings', 'woocommerce_zarinpalgateway_settings', 'woocommerce_WC_ZPal_settings'],
                    'keys' => ['merchantcode', 'merchant_id', 'merchantId', 'MerchantID', 'merchant', 'merchantCode'],
                ],
                'zibal' => [
                    'options' => ['woocommerce_zibal_settings', 'woocommerce_wc_zibal_settings', 'woocommerce_zibalgateway_settings'],
                    'keys' => ['merchant', 'merchant_id', 'merchantId', 'MerchantID', 'merchantcode', 'merchantCode'],
                ],
            ];
            foreach ($map as $gateway => $cfg) {
                if ($gateway === 'zarinpal' && $detected['zarinpal_merchant_id']) continue;
                if ($gateway === 'zibal' && $detected['zibal_merchant_id']) continue;
                foreach ($cfg['options'] as $optKey) {
                    $s = get_option($optKey, []);
                    if (!is_array($s) || (($s['enabled'] ?? 'yes') !== 'yes')) continue;
                    foreach ($cfg['keys'] as $k) {
                        $v = isset($s[$k]) ? trim((string)$s[$k]) : '';
                        if (!$v) continue;
                        $detected[$gateway === 'zarinpal' ? 'zarinpal_merchant_id' : 'zibal_merchant_id'] = $v;
                        break 2;
                    }
                }
            }
            $gatewayDefault = '';
            if ($defaultGateway) {
                if (str_contains($defaultGateway, 'zarin')) $gatewayDefault = 'zarinpal';
                if (str_contains($defaultGateway, 'zibal')) $gatewayDefault = 'zibal';
            }
            $next = array_merge($opts, [
                'zarinpal_merchant_id' => (string)$detected['zarinpal_merchant_id'],
                'zibal_merchant_id' => (string)$detected['zibal_merchant_id'],
                'payment_merchant_source' => 'auto',
            ]);
            if ($force || empty($next['payment_gateway_default'])) $next['payment_gateway_default'] = (string)$gatewayDefault;
            update_option(self::OPTION, self::sanitize($next), false);
            update_option(self::PAYMENT_SYNC_OPTION, ['last' => time()], false);
        } catch (\Throwable $e) {
        }
    }

    public static function maybePaymentBridge(): void
    {
        if (is_admin()) return;
        if (!isset($_GET['yektabot_pay']) || (string)$_GET['yektabot_pay'] !== '1') return;
        $canonical = function_exists('home_url') ? (string)home_url('/') : '';
        $canonHost = $canonical ? (string)parse_url($canonical, PHP_URL_HOST) : '';
        $currentHost = isset($_SERVER['HTTP_HOST']) ? (string)$_SERVER['HTTP_HOST'] : '';
        if ($canonHost && $currentHost && !hash_equals(strtolower($canonHost), strtolower($currentHost))) {
            $uri = isset($_SERVER['REQUEST_URI']) ? (string)$_SERVER['REQUEST_URI'] : '/';
            $to = rtrim($canonical, '/') . $uri;
            wp_redirect($to, 302);
            exit;
        }
        self::renderPaymentBridge();
        exit;
    }

    private static function renderPaymentBridge(): void
    {
        if (function_exists('nocache_headers')) nocache_headers();
        header('Content-Type: text/html; charset=utf-8');
        $token = isset($_GET['yb_token']) ? (string)$_GET['yb_token'] : '';
        if (!$token) {
            self::renderPayResult(false, 'پرداخت', 'لینک پرداخت نامعتبر است.');
            return;
        }
        $opts = self::opts();
        $secret = (string)($opts['secret'] ?? '');
        if (!$secret) {
            self::renderPayResult(false, 'پرداخت', 'اتصال افزونه به یکتابات انجام نشده است.');
            return;
        }
        [$payload, $payloadJson, $sig] = self::decodeSignedToken($token);
        if (!$payload || !$payloadJson || !$sig) {
            self::renderPayResult(false, 'پرداخت', 'لینک پرداخت نامعتبر است.');
            return;
        }
        $expected = hash_hmac('sha256', $payloadJson, $secret, true);
        if (!hash_equals($expected, $sig)) {
            self::renderPayResult(false, 'پرداخت', 'لینک پرداخت نامعتبر است.');
            return;
        }
        $exp = isset($payload['exp']) ? (int)$payload['exp'] : 0;
        if ($exp > 0 && (time() - 30) > $exp) {
            self::renderPayResult(false, 'پرداخت', 'این لینک منقضی شده است.');
            return;
        }
        $orderId = (int)($payload['order_id'] ?? 0);
        $amount = (int)($payload['amount'] ?? 0);
        $payloadAccount = (string)($payload['account_id'] ?? '');
        if ($orderId <= 0 || $amount <= 0) {
            self::renderPayResult(false, 'پرداخت', 'لینک پرداخت نامعتبر است.');
            return;
        }
        if (!empty($opts['account_id']) && $payloadAccount && !hash_equals((string)$opts['account_id'], $payloadAccount)) {
            self::renderPayResult(false, 'پرداخت', 'لینک پرداخت نامعتبر است.');
            return;
        }

        $gateway = self::resolvePaymentGateway($payload, $opts);
        if (!$gateway) {
            self::renderPayResult(false, 'پرداخت', 'لینک پرداخت نامعتبر است.');
            return;
        }
        $merchant = $gateway === 'zarinpal' ? trim((string)($opts['zarinpal_merchant_id'] ?? '')) : trim((string)($opts['zibal_merchant_id'] ?? ''));
        if (!$merchant) {
            self::renderPayResult(false, 'پرداخت', 'کلید درگاه پرداخت تنظیم نشده است.');
            return;
        }

        $callbackUrl = home_url('/?yektabot_pay=1&yb_token=' . rawurlencode($token));
        $amountRial = $amount * 10;

        $isCallback = $gateway === 'zarinpal'
            ? (isset($_GET['Authority']) || isset($_GET['Status']))
            : isset($_GET['trackId']);

        if (!$isCallback) {
            $trace = substr(hash('sha256', $token), 0, 16);
            $url = $gateway === 'zarinpal'
                ? self::zarinpalRequest($merchant, $amountRial, $callbackUrl, $orderId, (string)($payload['mobile'] ?? ''), $trace)
                : self::zibalRequest($merchant, $amountRial, $callbackUrl, $trace);
            if ($url) {
                wp_redirect($url);
                exit;
            }
            $hint = '';
            if (function_exists('get_transient')) {
                $last = get_transient('yektabot_pay_err_' . $trace);
                if (is_array($last) && !empty($last['message'])) $hint = ' (' . (string)$last['message'] . ')';
            }
            self::renderPayResult(false, 'پرداخت', 'خطا در اتصال به درگاه پرداخت. لطفاً دوباره تلاش کنید.' . $hint, $orderId, $amount);
            return;
        }

        $igUsername = (string)($payload['ig_username'] ?? ($opts['ig_username'] ?? ''));
        if ($gateway === 'zarinpal') {
            $authority = isset($_GET['Authority']) ? sanitize_text_field((string)$_GET['Authority']) : '';
            $status = isset($_GET['Status']) ? sanitize_text_field((string)$_GET['Status']) : '';
            if (!$authority || strtoupper($status) !== 'OK') {
                self::confirmPaymentToYektabot($token, 'cancelled', 'zarinpal', $authority ?: null, ['status' => $status ?: 'CANCELLED']);
                self::renderPayResult(false, 'پرداخت', 'پرداخت لغو شد.', $orderId, $amount, $igUsername);
                return;
            }
            $verified = self::zarinpalVerify($merchant, $authority, $amountRial);
            $ok = !empty($verified['ok']);
            $ref = isset($verified['ref']) ? (string)$verified['ref'] : $authority;
            self::confirmPaymentToYektabot($token, $ok ? 'paid' : 'failed', 'zarinpal', $ref ?: null, ['authority' => $authority, 'verify' => $verified]);
            self::renderPayResult($ok, 'پرداخت', $ok ? 'پرداخت با موفقیت انجام شد.' : 'پرداخت ناموفق بود. لطفاً دوباره تلاش کنید.', $orderId, $amount, $igUsername);
            return;
        }

        $trackId = isset($_GET['trackId']) ? sanitize_text_field((string)$_GET['trackId']) : '';
        if (!$trackId) {
            self::confirmPaymentToYektabot($token, 'cancelled', 'zibal', null, ['status' => 'CANCELLED']);
            self::renderPayResult(false, 'پرداخت', 'پرداخت لغو شد.', $orderId, $amount, $igUsername);
            return;
        }
        $verified = self::zibalVerify($merchant, $trackId);
        $ok = !empty($verified['ok']);
        self::confirmPaymentToYektabot($token, $ok ? 'paid' : 'failed', 'zibal', $trackId, ['trackId' => $trackId, 'verify' => $verified]);
        self::renderPayResult($ok, 'پرداخت', $ok ? 'پرداخت با موفقیت انجام شد.' : 'پرداخت ناموفق بود. لطفاً دوباره تلاش کنید.', $orderId, $amount, $igUsername);
    }

    private static function resolvePaymentGateway(array $payload, array $opts): ?string
    {
        if (empty($opts['payment_enabled'])) return null;
        $requested = isset($payload['gateway']) ? (string)$payload['gateway'] : '';
        if (in_array($requested, ['zarinpal', 'zibal'], true)) {
            $idKey = $requested === 'zarinpal' ? 'zarinpal_merchant_id' : 'zibal_merchant_id';
            if (!empty($opts[$idKey])) return $requested;
        }
        $default = isset($opts['payment_gateway_default']) ? (string)$opts['payment_gateway_default'] : '';
        if (in_array($default, ['zarinpal', 'zibal'], true)) {
            $idKey = $default === 'zarinpal' ? 'zarinpal_merchant_id' : 'zibal_merchant_id';
            if (!empty($opts[$idKey])) return $default;
        }
        if (!empty($opts['zarinpal_merchant_id'])) return 'zarinpal';
        if (!empty($opts['zibal_merchant_id'])) return 'zibal';
        return null;
    }

    private static function connectionStatus(): ?bool
    {
        if (!function_exists('get_transient') || !function_exists('set_transient') || !function_exists('wp_remote_post') || !function_exists('wp_remote_retrieve_response_code') || !function_exists('wp_json_encode')) return null;
        $opts = self::opts();
        if (empty($opts['url']) || empty($opts['secret']) || empty($opts['account_id'])) return null;
        $cache = get_transient(self::CONNECTION_OK_TRANSIENT);
        if ($cache === '1') return true;
        if ($cache === '0') return false;
        $body = wp_json_encode(['site' => home_url('/'), 'version' => self::VERSION, 'install_id' => self::installId()]);
        if (!$body) return null;
        $headers = ['Accept' => 'application/json', 'Content-Type' => 'application/json', 'X-Yektabot-Signature' => base64_encode(hash_hmac('sha256', (string)$body, (string)$opts['secret'], true))];
        $resp = wp_remote_post(self::REQUIRE_URL, ['timeout' => 6, 'headers' => $headers, 'body' => $body]);
        if (is_wp_error($resp)) return null;
        $code = (int)wp_remote_retrieve_response_code($resp);
        if ($code === 200) {
            set_transient(self::CONNECTION_OK_TRANSIENT, '1', MINUTE_IN_SECONDS * 2);
            return true;
        }
        if (in_array($code, [401, 403, 404], true)) {
            set_transient(self::CONNECTION_OK_TRANSIENT, '0', MINUTE_IN_SECONDS * 2);
            return false;
        }
        return null;
    }

    private static function disconnect(): void
    {
        $opts = self::opts();
        self::track('disconnected', []);
        self::cleanup();
        delete_option(self::OPTION);
        delete_option(self::CONNECT_OPTION);
        delete_option(self::FULL_SYNC_OPTION);
        delete_option(self::PAYMENT_SYNC_OPTION);
        delete_option(self::TRACK_OPTION);
        delete_option(self::INSTALL_ID_OPTION);
        delete_transient(self::CONNECTION_OK_TRANSIENT);
        wp_safe_redirect(admin_url('admin.php?page=' . self::PAGE . '&disconnected=1'));
        exit;
    }

    private static function switchHtml(string $name, bool $checked, string $label): string
    {
        static $i = 0;
        $i++;
        $id = 'yb-' . preg_replace('/[^a-z0-9_\-]/i', '', $name) . '-' . $i;
        return '<label class="yb-switch"><input id="' . esc_attr($id) . '" type="checkbox" name="' . esc_attr($name) . '" value="1"' . ($checked ? ' checked' : '') . '/><span class="yb-switch-ui"></span><span>' . esc_html($label) . '</span></label>';
    }

    private static function confirmPaymentToYektabot(string $token, string $status, string $gateway, ?string $ref, array $meta): void
    {
        try {
            if (!function_exists('wp_remote_post') || !function_exists('wp_json_encode')) return;
            $opts = self::opts();
            if (empty($opts['secret'])) return;
            $body = wp_json_encode([
                'token' => $token,
                'status' => $status,
                'gateway' => $gateway,
                'ref' => $ref,
                'meta' => $meta,
            ]);
            if (!$body) return;
            $sig = base64_encode(hash_hmac('sha256', $body, (string)$opts['secret'], true));
            wp_remote_post(self::PAYMENT_CONFIRM_URL, [
                'timeout' => 15,
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'X-Yektabot-Signature' => $sig,
                ],
                'body' => $body,
            ]);
        } catch (\Throwable $e) {
        }
    }

    private static function zarinpalRequest(string $merchantId, int $amountRial, string $callbackUrl, int $orderId, string $mobile, string $trace = ''): ?string
    {
        if (!function_exists('wp_remote_post') || !function_exists('wp_json_encode') || !function_exists('wp_remote_retrieve_body')) return null;
        $body = wp_json_encode([
            'merchant_id' => $merchantId,
            'amount' => $amountRial,
            'callback_url' => $callbackUrl,
            'description' => 'payment for order: ' . $orderId,
            'metadata' => array_filter(['email' => null, 'mobile' => $mobile ?: null], fn($v) => $v !== null && $v !== ''),
        ]);
        if (!$body) return null;
        $res = wp_remote_post('https://api.zarinpal.com/pg/v4/payment/request.json', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json', 'Accept' => 'application/json'],
            'body' => $body,
        ]);
        if (is_wp_error($res)) {
            self::log('error', 'payment_gateway_request_failed', ['gateway' => 'zarinpal', 'order_id' => $orderId, 'amount_rial' => $amountRial, 'error' => $res->get_error_message()]);
            if ($trace && function_exists('set_transient')) set_transient('yektabot_pay_err_' . $trace, ['message' => $res->get_error_message()], 90);
            return null;
        }
        $http = function_exists('wp_remote_retrieve_response_code') ? (int)wp_remote_retrieve_response_code($res) : null;
        $json = json_decode((string)wp_remote_retrieve_body($res), true);
        $code = is_array($json) ? (int)($json['data']['code'] ?? 0) : 0;
        $authority = is_array($json) ? (string)($json['data']['authority'] ?? '') : '';
        if ($code !== 100 || !$authority) {
            $err = is_array($json) ? ($json['errors'] ?? null) : null;
            self::log('error', 'payment_gateway_request_failed', ['gateway' => 'zarinpal', 'order_id' => $orderId, 'amount_rial' => $amountRial, 'http' => $http, 'code' => $code, 'errors' => $err]);
            $msg = is_array($err) ? (string)($err['message'] ?? '') : '';
            if ($msg && str_contains($msg, 'banned')) $msg = 'دامنه/ترمینال زرین‌پال اجازه تراکنش ندارد';
            if ($trace && function_exists('set_transient')) set_transient('yektabot_pay_err_' . $trace, array_filter(['message' => $msg ?: null, 'code' => is_array($err) ? ($err['code'] ?? null) : null, 'http' => $http], fn($v) => $v !== null && $v !== ''), 90);
            return null;
        }
        return 'https://www.zarinpal.com/pg/StartPay/' . $authority;
    }

    private static function zarinpalVerify(string $merchantId, string $authority, int $amountRial): array
    {
        if (!function_exists('wp_remote_post') || !function_exists('wp_json_encode') || !function_exists('wp_remote_retrieve_body')) return ['ok' => false];
        $body = wp_json_encode(['merchant_id' => $merchantId, 'authority' => $authority, 'amount' => $amountRial]);
        if (!$body) return ['ok' => false];
        $res = wp_remote_post('https://api.zarinpal.com/pg/v4/payment/verify.json', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json', 'Accept' => 'application/json'],
            'body' => $body,
        ]);
        if (is_wp_error($res)) {
            self::log('error', 'payment_gateway_verify_failed', ['gateway' => 'zarinpal', 'authority' => $authority, 'amount_rial' => $amountRial, 'error' => $res->get_error_message()]);
            return ['ok' => false, 'error' => $res->get_error_message()];
        }
        $http = function_exists('wp_remote_retrieve_response_code') ? (int)wp_remote_retrieve_response_code($res) : null;
        $json = json_decode((string)wp_remote_retrieve_body($res), true);
        $code = is_array($json) ? (int)($json['data']['code'] ?? 0) : 0;
        $ref = is_array($json) ? (string)($json['data']['ref_id'] ?? '') : '';
        if (!in_array($code, [100, 101], true)) self::log('error', 'payment_gateway_verify_failed', ['gateway' => 'zarinpal', 'authority' => $authority, 'amount_rial' => $amountRial, 'http' => $http, 'code' => $code, 'raw' => $json]);
        return ['ok' => in_array($code, [100, 101], true), 'code' => $code, 'ref' => $ref, 'raw' => $json];
    }

    private static function zibalRequest(string $merchant, int $amountRial, string $callbackUrl, string $trace = ''): ?string
    {
        if (!function_exists('wp_remote_post') || !function_exists('wp_json_encode') || !function_exists('wp_remote_retrieve_body')) return null;
        $body = wp_json_encode(['merchant' => $merchant, 'callbackUrl' => $callbackUrl, 'amount' => $amountRial]);
        if (!$body) return null;
        $res = wp_remote_post('https://gateway.zibal.ir/v1/request', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json', 'Accept' => 'application/json'],
            'body' => $body,
        ]);
        if (is_wp_error($res)) {
            self::log('error', 'payment_gateway_request_failed', ['gateway' => 'zibal', 'amount_rial' => $amountRial, 'error' => $res->get_error_message()]);
            if ($trace && function_exists('set_transient')) set_transient('yektabot_pay_err_' . $trace, ['message' => $res->get_error_message()], 90);
            return null;
        }
        $http = function_exists('wp_remote_retrieve_response_code') ? (int)wp_remote_retrieve_response_code($res) : null;
        $json = json_decode((string)wp_remote_retrieve_body($res));
        $result = isset($json->result) ? (int)$json->result : 0;
        $trackId = isset($json->trackId) ? (string)$json->trackId : '';
        if ($result !== 100 || !$trackId) {
            self::log('error', 'payment_gateway_request_failed', ['gateway' => 'zibal', 'amount_rial' => $amountRial, 'http' => $http, 'result' => $result, 'message' => isset($json->message) ? (string)$json->message : null]);
            $msg = isset($json->message) ? trim((string)$json->message) : '';
            if ($trace && function_exists('set_transient')) set_transient('yektabot_pay_err_' . $trace, array_filter(['message' => $msg ?: null, 'result' => $result, 'http' => $http], fn($v) => $v !== null && $v !== ''), 90);
            return null;
        }
        return 'https://gateway.zibal.ir/start/' . $trackId;
    }

    private static function zibalVerify(string $merchant, string $trackId): array
    {
        if (!function_exists('wp_remote_post') || !function_exists('wp_json_encode') || !function_exists('wp_remote_retrieve_body')) return ['ok' => false];
        $body = wp_json_encode(['merchant' => $merchant, 'trackId' => $trackId]);
        if (!$body) return ['ok' => false];
        $res = wp_remote_post('https://gateway.zibal.ir/v1/verify', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json', 'Accept' => 'application/json'],
            'body' => $body,
        ]);
        if (is_wp_error($res)) {
            self::log('error', 'payment_gateway_verify_failed', ['gateway' => 'zibal', 'trackId' => $trackId, 'error' => $res->get_error_message()]);
            return ['ok' => false, 'error' => $res->get_error_message()];
        }
        $http = function_exists('wp_remote_retrieve_response_code') ? (int)wp_remote_retrieve_response_code($res) : null;
        $json = json_decode((string)wp_remote_retrieve_body($res), true);
        $result = is_array($json) ? (int)($json['result'] ?? 0) : 0;
        if ($result !== 100) self::log('error', 'payment_gateway_verify_failed', ['gateway' => 'zibal', 'trackId' => $trackId, 'http' => $http, 'result' => $result, 'raw' => $json]);
        return ['ok' => $result === 100, 'result' => $result, 'raw' => $json];
    }

    private static function decodeSignedToken(string $token): array
    {
        $parts = explode('.', $token, 2);
        if (count($parts) !== 2) return [null, null, null];
        $payloadJson = self::b64urlDecode($parts[0]);
        $sig = self::b64urlDecode($parts[1]);
        $payload = $payloadJson ? json_decode($payloadJson, true) : null;
        if (!is_array($payload) || !$payloadJson || !$sig) return [null, null, null];
        return [$payload, $payloadJson, $sig];
    }

    private static function b64urlDecode(string $value): string
    {
        $v = trim($value);
        if ($v === '') return '';
        $v = strtr($v, '-_', '+/');
        $pad = strlen($v) % 4;
        if ($pad) $v .= str_repeat('=', 4 - $pad);
        $decoded = base64_decode($v, true);
        return $decoded === false ? '' : $decoded;
    }

    private static function renderPayResult(bool $ok, string $title, string $message, ?int $orderId = null, ?int $amount = null, ?string $igUsername = null): void
    {
        $fontUrl = self::fontUrl();
        $fontFace = $fontUrl ? '@font-face{font-family:YekanBakhFaNum_v;src:url("' . esc_url($fontUrl) . '") format("woff-variation"),url("' . esc_url($fontUrl) . '") format("woff");font-display:swap;font-weight:100 900;font-style:normal}' : '';
        $bg = $ok ? '#ecfdf5' : '#fff7ed';
        $accent = $ok ? '#16a34a' : '#ea580c';
        $orderLine = $orderId ? ('<div class="row"><b>شماره سفارش</b><span>' . esc_html((string)$orderId) . '</span></div>') : '';
        $amountLine = $amount ? ('<div class="row"><b>مبلغ</b><span>' . esc_html(number_format($amount)) . ' تومان</span></div>') : '';
        $igLine = $igUsername ? ('<a class="ig" href="https://instagram.com/' . esc_attr(ltrim($igUsername, '@')) . '" target="_blank" rel="noopener">بازگشت به اینستاگرام</a>') : '';
        echo '<!doctype html><html lang="fa" dir="rtl"><head><meta charset="utf-8"/><meta name="viewport" content="width=device-width,initial-scale=1"/><title>' . esc_html($title) . '</title><style>' . $fontFace . 'body{margin:0;background:#f7f8fa;color:#111827;font-family:YekanBakhFaNum_v,IRANSans,system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif}*{box-sizing:border-box}a{text-decoration:none}main{max-width:720px;margin:40px auto;padding:0 16px}section{background:#fff;border:1px solid #e5e7eb;border-radius:18px;box-shadow:0 18px 45px rgba(0,0,0,.06);overflow:hidden}header{padding:18px 20px;background:' . $bg . ';border-bottom:1px solid #e5e7eb}h1{margin:0;font-size:20px;font-weight:900;color:#111827}p{margin:8px 0 0;font-size:13px;font-weight:800;color:#374151}article{padding:18px 20px}article .row{display:flex;align-items:center;justify-content:space-between;gap:14px;padding:10px 12px;border-radius:14px;background:#f3f4f6;margin-bottom:10px}article .row b{font-size:13px;font-weight:900}article .row span{font-size:12px;font-weight:900;color:#374151}footer{padding:16px 20px;border-top:1px solid #e5e7eb;display:flex;align-items:center;justify-content:space-between;gap:12px}footer .badge{display:inline-flex;align-items:center;justify-content:center;min-width:60px;height:32px;border-radius:999px;background:' . $bg . ';color:' . $accent . ';font-size:12px;font-weight:900;padding:0 12px}footer .ig{display:inline-flex;align-items:center;justify-content:center;height:38px;border-radius:14px;padding:0 14px;background:#111827;color:#fff;font-weight:900;font-size:12px}</style></head><body><main><section><header><h1>' . esc_html($title) . '</h1><p>' . esc_html($message) . '</p></header><article>' . $orderLine . $amountLine . '</article><footer><span class="badge">' . ($ok ? 'موفق' : 'ناموفق') . '</span>' . $igLine . '</footer></section></main></body></html>';
    }

    public static function maybeSync(): void
    {
        if (!is_admin() || !current_user_can('manage_options')) return;
        if (get_option(self::OPTION . '_synced') === (string)self::signature()) return;
        if (!self::ready(true)) return;
        self::sync(true);
    }

    public static function maybeTrack(): void
    {
        if (!is_admin() || !current_user_can('manage_options')) return;
        $state = get_option(self::TRACK_OPTION, []);
        $state = is_array($state) ? $state : [];
        if (empty($state['id'])) {
            $state['id'] = function_exists('random_bytes') ? bin2hex(random_bytes(16)) : (function_exists('wp_generate_password') ? wp_generate_password(32, false) : md5(uniqid('', true)));
            $state['created_at'] = time();
        }
        $last = (int)($state['last'] ?? 0);
        if ($last > 0 && (time() - $last) < 43200) return;
        $state['last'] = time();
        update_option(self::TRACK_OPTION, $state, false);
        self::track('heartbeat', []);
    }

    public static function activate(): void
    {
        self::track('activated', []);
        self::sync();
    }

    public static function deactivate(): void
    {
        self::track('deactivated', []);
        self::cleanup();
    }

    public static function cleanup(): void
    {
        self::loadWoo();
        $opts = self::opts();
        $url = $opts['url'] ? self::target($opts) : '';
        $table = self::webhookTable();
        if ($table) {
            global $wpdb;
            $like = $wpdb->esc_like('Yektabot ') . '%';
            if ($url) $wpdb->query($wpdb->prepare("DELETE FROM {$table} WHERE name LIKE %s AND delivery_url = %s", $like, $url));
            else $wpdb->query($wpdb->prepare("DELETE FROM {$table} WHERE name LIKE %s", $like));
        }
        if (class_exists('\WC_Webhooks') && method_exists('\WC_Webhooks', 'get_webhooks')) {
            foreach (self::webhooks() as $hook) {
                if ($hook->get_name() === 'Yektabot ' . $hook->get_topic() && ($url ? $hook->get_delivery_url() === $url : true)) $hook->delete(true);
            }
        } else {
            $ids = get_posts(['post_type' => 'shop_webhook', 'post_status' => 'any', 'posts_per_page' => -1, 'fields' => 'ids']);
            foreach ($ids as $id) {
                $title = (string)get_the_title((int)$id);
                if (!str_starts_with($title, 'Yektabot ')) continue;
                if ($url) {
                    $delivery = (string)(get_post_meta((int)$id, '_webhook_delivery_url', true) ?: get_post_meta((int)$id, 'webhook_delivery_url', true));
                    if ($delivery !== $url) continue;
                }
                wp_delete_post((int)$id, true);
            }
        }
        if (!empty($opts['api_key_id']) && is_numeric($opts['api_key_id'])) {
            global $wpdb;
            if (isset($wpdb) && property_exists($wpdb, 'prefix')) {
                $keysTable = $wpdb->prefix . 'woocommerce_api_keys';
                $wpdb->query($wpdb->prepare("DELETE FROM {$keysTable} WHERE key_id = %d", (int)$opts['api_key_id']));
            }
        }
        delete_option(self::OPTION . '_synced');
        self::log('info', 'cleanup', ['url' => $url]);
    }

    private static function track(string $event, array $extra): void
    {
        try {
            if (!function_exists('wp_remote_post') || !function_exists('wp_json_encode')) return;
            $installId = self::installId();
            if (!$installId) return;
            $opts = self::opts();
            $payload = array_merge([
                'event' => $event,
                'install_id' => $installId,
                'plugin_version' => self::VERSION,
                'wp_version' => function_exists('get_bloginfo') ? (string)get_bloginfo('version') : '',
                'php_version' => defined('PHP_VERSION') ? (string)PHP_VERSION : '',
                'wc_version' => defined('WC_VERSION') ? (string)WC_VERSION : '',
            ], $extra);
            if (empty($payload['account_id']) && !empty($opts['account_id'])) $payload['account_id'] = (int)$opts['account_id'];
            if (empty($payload['ig_username']) && !empty($opts['ig_username'])) $payload['ig_username'] = (string)$opts['ig_username'];
            if (empty($payload['ig_name']) && !empty($opts['ig_name'])) $payload['ig_name'] = (string)$opts['ig_name'];
            if (empty($payload['site']) && !empty($opts['account_id'])) $payload['site'] = home_url('/');
            if (empty($payload['payment_url'])) $payload['payment_url'] = home_url('/?yektabot_pay=1');
            if (empty($payload['payment_gateway_default']) && !empty($opts['payment_gateway_default'])) $payload['payment_gateway_default'] = (string)$opts['payment_gateway_default'];
            if (empty($payload['payment_enabled'])) $payload['payment_enabled'] = !empty($opts['payment_enabled']);
            if (!array_key_exists('payment_gateways', $payload)) {
                $gateways = [];
                if (!empty($opts['payment_enabled'])) {
                    if (!empty($opts['zarinpal_merchant_id'])) $gateways[] = 'zarinpal';
                    if (!empty($opts['zibal_merchant_id'])) $gateways[] = 'zibal';
                }
                $payload['payment_gateways'] = $gateways;
            }
            $json = wp_json_encode(array_filter($payload, fn($v) => $v !== null && $v !== ''));
            if (!$json) return;
            $headers = ['Content-Type' => 'application/json'];
            if (!empty($opts['secret'])) $headers['X-Yektabot-Signature'] = base64_encode(hash_hmac('sha256', $json, (string)$opts['secret'], true));
            wp_remote_post(self::TRACK_URL, [
                'timeout' => 6,
                'blocking' => false,
                'headers' => $headers,
                'body' => $json,
            ]);
        } catch (\Throwable $e) {
        }
    }
}

Plugin::boot();


